"""
SPU register exception
"""
class SPURegException(Exception):
	def __init__(self, value):
		self.value = value
	def __str__(self):
		return repr(self.value)

"""
SPU register
"""
class SPUReg:	
	def __init__(self, sreg1 = 0x00000000, sreg2 = 0x00000000, sreg3 = 0x00000000, sreg4 = 0x00000000):
		self.value = [sreg1, sreg2, sreg3, sreg4]
	
	def __repr__(self):
		return "(%08X %08X %08X %08X)" % (self[0], self[1], self[2], self[3])
	
	def __getitem__(self, idx):
		if idx < 0 or idx > 3:
			raise SPURegException("Subreg %d does not exist!" % idx)
		return self.value[idx]
	
	def __setitem__(self, idx, v):
		if idx < 0 or idx > 3:
			raise SPURegException("Subreg %d does not exist!" % idx)
		self.value[idx] = v
	
	def to_bytes(self):
		res = []
		for i in range(4):
			for j in range(4):
				v = self[i] >> (24 - j * 8)
				res.append(v & 0xFF)
		return res
	
	def from_bytes(self, bytes):
		for i in range(4):
			self[i] = 0
			for j in range(4):
				self[i] |= bytes.pop(0) << (24 - j * 8)

"""
SPU instructions emulator exception
"""
class SPUInstEmuException(Exception):
	def __init__(self, value):
		self.value = value
	def __str__(self):
		return repr(self.value)

"""
SPU instructions emulator
"""
class SPUInstEmu:
	IW = 0
	IH = 1
	IB = 3
	
	def __init__(self):
		self._NUM_REGS = 128
		self.regs = []
		for i in range(self._NUM_REGS):
			self.regs.append(SPUReg())
	
	def print_reg(self, idx):
		if idx < 0 or idx > len(self.regs):
			raise SPUInstEmuException("Register %d does not exist" % idx)
		print(" %03d : %s" % (idx, self.regs[idx]))
	
	def print_regs(self):
		print("SPU Registers:")
		for i in range(len(self.regs)):
			self.print_reg(i)
	
	def inst_rotm(self, rt, ra, rb):
		for i in range(4):
			shift_count = (-self.regs[rb][i]) & 63
			if shift_count < 32:
				self.regs[rt][i] = self.regs[ra][i] >> shift_count
			else:
				self.regs[rt][i] = 0
	
	def inst_shlqbyi(self, rt, ra, i7):
		rtb = self.regs[rt].to_bytes()
		rab = self.regs[ra].to_bytes()
		i7 &= 0x1F;
		for i in range(16):
			rtb[i] = 0x00 if (i + i7) >= 16 else rab[i + i7]
		self.regs[rt].from_bytes(rtb)
	
	def inst_fsmbi(self, rt, i16):
		rtb = self.regs[rt].to_bytes()
		for i in range(16):
			rtb[i] = 0xFF if (i16 & (1<<(15-i))) else 0x00
		self.regs[rt].from_bytes(rtb)
	
	def inst_andbi(self, rt, ra, i10):
		rtb = self.regs[rt].to_bytes()
		rab = self.regs[ra].to_bytes()
		for i in range(16):
			rtb[i] = rab[i] & i10
		self.regs[rt].from_bytes(rtb)
	
	def inst_cg(self, rt, ra, rb):
		for i in range(4):
			self.regs[rt][i] = 0x00000001 if ((self.regs[ra][i] + self.regs[rb][i]) < self.regs[ra][i]) else 0x00000000

if __name__ == "__main__":
	a = SPUInstEmu()
	a.regs[89] = SPUReg(sreg2=0x89abcdef)
	a.inst_fsmbi(35, 0x101)
	a.inst_andbi(33, 35, 0x20)
	a.inst_cg(34, 89, 33)
	a.inst_shlqbyi(4, 89, 4)
	print(a.regs[35])
	print(a.regs[33])
	print(a.regs[34])
	print(a.regs[4])
