// Copyright 2011 naehrwert
// Licensed under the terms of the GNU GPL, version 2
// http://www.gnu.org/licenses/old-licenses/gpl-2.0.txt

#include "common.h"
#include "config.h"
#include "disasm.h"
#include "spud.h"
#include "subroutine.h"
#include "block.h"

static subroutine_t *_subroutine_copy(subroutine_t *sr)
{
	subroutine_t *res = new subroutine_t;
	memcpy(res, sr, sizeof(subroutine_t));
	return res;
}

static void _subroutine_split(ctxt_t *ctxt, u32 target, list<subroutine_t *>::iterator &it)
{
	subroutine_t *tsr = *it;

	//Create two new subroutines.
	subroutine_t *first = _subroutine_copy(tsr);
	subroutine_t *second = _subroutine_copy(tsr);
	
	//First subroutine ends with the instruction previous the target one.
	instr_t *inst = disasm_get_instr(tsr->er, target - INSTR_SIZE);
	//Ignore nop/lnop at the end of the first subroutine.
	if(inst->instr == INSTR_NOP || inst->instr == INSTR_LNOP)
		inst = disasm_get_instr(tsr->er, target - INSTR_SIZE*2);
	first->eidx = inst->idx;
	
	//Second subroutine starts with target instruction.
	second->sidx = disasm_get_instr(tsr->er, target)->idx;

	//Replace the subroutines in the original list.
	ctxt->subroutines.insert(it, first);
	ctxt->subroutines.insert(it, second);
	ctxt->subroutines.erase(it);

	DBGPRINTF("subroutine: split sub @ 0x%05x into sub @ 0x%05x and sub @ 0x%05x\n",
		SUBSADDR(tsr), SUBSADDR(first), SUBSADDR(second));

	//Delete the splitted subroutine.
	//XXX: Better not yet.
	//delete tsr;
}

static void _subroutine_split_all(ctxt_t *ctxt)
{
	//Outer and inner iterators.
	list<subroutine_t *>::iterator oit, iit;

	for(oit = ctxt->subroutines.begin(); oit != ctxt->subroutines.end(); ++oit)
	{
		subroutine_t *sr = *oit;
		int i;
		u32 csaddr = SUBSADDR(sr);
		u32 ceaddr = SUBEADDR(sr);

		//Check for branch with a target inside another subroutine.
		for(i = sr->sidx; i <= sr->eidx; i++)
		{
			instr_t *inst = &(sr->er->instrs[i]);
			if(disasm_is_direct_branch(inst))
			{
				u32 bt = BRANCH_TARGET(inst);
				//Make sure that this is not a branch inside the current subroutine.
				if(!(bt >= csaddr && bt <= ceaddr))
				{
					for(iit = ctxt->subroutines.begin(); iit != ctxt->subroutines.end(); ++iit)
					{
						subroutine_t *tsr = *iit;
						//Not to the start, because then it's just a function call.
						if(bt > SUBSADDR(tsr) && bt <= SUBEADDR(tsr))
						{
							//Split target subroutine.
							_subroutine_split(ctxt, bt, iit);
							break;
						}
					}
				}
			}
		}
	}
}

static void _subroutine_find_refs(ctxt_t *ctxt, subroutine_t *sr)
{
	unsigned int i, j;
	list<subroutine_t *>::iterator it;

	//Check all subroutines.
	for(it = ctxt->subroutines.begin(); it != ctxt->subroutines.end(); ++it)
	{
		subroutine_t *tsr = *it;
		//Check all instructions.
		for(j = tsr->sidx; j <= tsr->eidx; j++)
		{
			instr_t *inst = &(tsr->er->instrs[j]);
			//Check for a relative branch to start of subroutine.
			if(disasm_is_direct_branch(inst) && BRANCH_TARGET(inst) == SUBSADDR(sr))
			{
				DBGPRINTF("subroutine: found ref from sub @ 0x%05x (instr @ 0x%05x) to 0x%05x\n", 
					SUBSADDR(tsr), IIDX2ADDR(tsr->er, j), SUBSADDR(sr));
				//Add reference to referenced subroutine.
				reference_t *ref = new reference_t;
				ref->subroutine = tsr;
				ref->refidx = j;
				sr->fsubrefs->push_back(ref);
				//Add reference to referencing subroutine.
				ref = new reference_t;
				ref->subroutine = sr;
				ref->refidx = sr->sidx;
				tsr->tsubrefs->push_back(ref);
			}
		}
	}
}

static void _subroutine_find_refs_all(ctxt_t *ctxt)
{
	int i;
	list<subroutine_t *>::iterator it;

	//Find references for all subroutines.
	for(it = ctxt->subroutines.begin(); it != ctxt->subroutines.end(); ++it)
		_subroutine_find_refs(ctxt, *it);
}

static void _subroutine_init(ctxt_t *ctxt)
{
	int i;
	list<subroutine_t *>::iterator it;

	for(it = ctxt->subroutines.begin(); it != ctxt->subroutines.end(); ++it)
	{
		subroutine_t *sr = *it;

		//Check for stop instruction instead of return.
		instr_t *inst = &(sr->er->instrs[sr->eidx]);
		if(inst->instr == INSTR_STOP)
			sr->noreturn = true;
		else
			sr->noreturn = false;

		sr->fsubrefs = new vector<reference_t *>();
		sr->tsubrefs = new vector<reference_t *>();
		sr->blocks = new list<block_t *>();
		sr->dfsblocks = new list<block_t *>();
		sr->revdfsblocks = new list<block_t *>();
		sr->edges = new vector<edge_t *>();
	}
}

static subroutine_t *_subroutine_extract(execr_t *er, unsigned int sidx)
{
	unsigned int i;

	for(i = sidx; i < er->instrs.size(); i++)
	{
		instr_t *inst = &(er->instrs[i]);
		//Try to find a bi $lr instruction for now.
		//A subroutine could also have more than one bi $lr, check this later.
		if(IS_RETURN(inst) /*|| inst->instr == INSTR_STOP*/)
		{
			subroutine_t *res = new subroutine_t;
			res->er = er;
			res->sidx = sidx;
			res->eidx = i;

			return res;
		}
		//Discard subroutines with unknown instructions.
		if(inst->instr == INSTR_NONE)
		{
			DBGPRINTF("subroutine: unknown instruction @ 0x%05x\n", IIDX2ADDR(er, i));
			return NULL;
		}
	}

	return NULL;
}

subroutine_t *subroutine_find_tsubref(subroutine_t *sr, u32 addr)
{
	int i;

	for(i = 0; i < sr->tsubrefs->size(); i++)
	{
		reference_t *ref = (*sr->tsubrefs)[i];
		//Check if the address matches a subroutine reference.
		if(IIDX2ADDR(ref->subroutine->er, ref->refidx) == addr)
			return ref->subroutine;
	}

	return NULL;
}

void subroutine_extract_all(ctxt_t *ctxt)
{
	unsigned int i, j;
	list<subroutine_t *>::iterator it;

	//Check all executable ranges.
	for(i = 0; i < ctxt->execrs.size(); i++)
	{
		execr_t *er = ctxt->execrs[i];
		j = 0;
		//Check all instructions.
		while(j < er->instrs.size())
		{
			instr_t *inst = &(er->instrs[j]);
			//Ignore nops and unknown instructions.
			if(inst->instr != INSTR_NOP && 
				inst->instr != INSTR_LNOP &&
				inst->instr != INSTR_NONE)
			{
				//Extract next subroutine.
				subroutine_t *sr = _subroutine_extract(er, j);
				if(sr != NULL)
				{
					DBGPRINTF("subroutine: found sub @ 0x%05x (end @ 0x%05x)\n", SUBSADDR(sr), SUBEADDR(sr));
					ctxt->subroutines.push_back(sr);
					//Move instruction index to subroutine end.
					j = sr->eidx;
				}
			}
			else
				DBGPRINTF("subroutine: skipped nop/lnop @ 0x%05x\n", IIDX2ADDR(er, inst->idx));
			j++;
		}
	}

	//Split subroutines.
	_subroutine_split_all(ctxt);

	//Create all lists and check if it's a noreturn subroutine.
	_subroutine_init(ctxt);

	//Now find all references.
	_subroutine_find_refs_all(ctxt);

	//Check if every subroutine is reachable and mark them respective.
	for(it = ctxt->subroutines.begin(); it != ctxt->subroutines.end(); ++it)
	{
		subroutine_t *sr = *it;
		sr->reachable = (sr->fsubrefs->size() == 0 && 
			SUBSADDR(sr) != ctxt->entry ? false : true);
		DBGPRINTF("subroutine: sub @ 0x%05x is %s\n", 
			SUBSADDR(sr), (sr->reachable == true ? "reachable" : "not reachable"));
	}
}
