// Copyright 2011 naehrwert
// Licensed under the terms of the GNU GPL, version 2
// http://www.gnu.org/licenses/old-licenses/gpl-2.0.txt

#ifndef _SPUD_H_
#define _SPUD_H_

#include <vector>
using std::vector;
#include <list>
using std::list;

#include "types.h"
#include "util.h"
#include "disasm.h"

//Executable range.
typedef struct _execr_t
{
	//Owning context.
	struct _ctxt_t *ctxt;
	//Start address.
	u32 start;
	//Size.
	u32 size;
	//Disassembled instructions.
	vector<instr_t> instrs;
} execr_t;

//Subroutine reference.
typedef struct _reference_t
{
	//Referencing subroutine.
	struct _subroutine_t *subroutine;
	//Referencing instruction index.
	unsigned int refidx;
} reference_t;

//Edge type.
//Directly go to the next node.
#define EDGE_BRANCH 0
//Go to the next node under some condition.
#define EDGE_BRANCH_COND 1
//Exit.
#define EDGE_EXIT 2
//Next.
#define EDGE_NEXT 3
//Error.
#define EDGE_ERROR 4

//Graph edge.
typedef struct _edge_t
{
	//Edge type.
	int type;
	//From node.
	struct _block_t *from;
	//To node.
	struct _block_t *to;
} edge_t;

//Block types.
//Start block.
#define BLOCK_START 0
//Simple block.
#define BLOCK_SIMPLE 1
//Call block.
#define BLOCK_CALL 2
//End block.
#define BLOCK_END 3

//Block node.
typedef struct _block_node_t
{
	int dfsnum;
	int dom_dfsnum_first;
	int dom_dfsnum_last;

	struct _block_node_t *dominator;
	struct _block_node_t *parent;
	
	//Children.
	list<struct _block_node_t *> *children;
	//Dominator children.
	list<struct _block_node_t *> *domchildren;
	//Frontier.
	list<struct _block_node_t *> *frontier;

	//Owning block.
	struct _block_t *block;
} block_node_t;

//Block.
typedef struct _block_t
{
	//Owning subroutine.
	struct _subroutine_t *sr;
	//Start instruction index.
	unsigned int sidx;
	//End instruction index.
	unsigned int eidx;
	//Exit instruction.
	instr_t *exitinst;
	//Type.
	int type;
	//In edges.
	vector<edge_t *> *inedge;
	//Out edges.
	vector<edge_t *> *outedge;

	block_node_t node;
	block_node_t revnode;
} block_t;

//Operation types.
//Move operation.
#define OP_TYPE_MOVE 1
//Arithmetical operation.
#define OP_TYPE_ARITH 2

typedef struct _operation_t
{
	//...
	//int type;
} operation_t;

//Subroutine.
typedef struct _subroutine_t
{
	//Owning executable range.
	execr_t *er;
	//Start instruction index;
	unsigned int sidx;
	//End instruction index.
	unsigned int eidx;
	//Reachable?
	bool reachable;
	//No return?
	bool noreturn;
	//References from other subroutines.
	vector<reference_t *> *fsubrefs;
	//Reference to other subroutines.
	vector<reference_t *> *tsubrefs;
	//Blocks.
	list<block_t *> *blocks;
	//DFS blocks.
	list<block_t *> *dfsblocks;
	//Reverse DFS blocks.
	list<block_t *> *revdfsblocks;
	//Edges.
	vector<edge_t *> *edges;

	//Used registers.
#define USED_REGS_SIZE BITMAP_SIZE(REG_COUNT)
	bitmap_t used_regs[USED_REGS_SIZE];

	//Temporary DFS number.
	int tmp;
} subroutine_t;

//Decompiler context.
typedef struct _ctxt_t
{
	//Local storage.
	u8 *ls;
	//Entry.
	u32 entry;
	//Executable ranges.
	vector<execr_t *> execrs;
	//Subroutines.
	list<subroutine_t *> subroutines;
} ctxt_t;

ctxt_t *spud_create_ctxt(const char *elf);
void spud_destroy_ctxt(ctxt_t *ctxt);
void spud_decompile(ctxt_t *ctxt);

#endif
