// Copyright 2011 naehrwert
// Copyright 2010 fail0verflow <master@fail0verflow.com>
// Licensed under the terms of the GNU GPL, version 2
// http://www.gnu.org/licenses/old-licenses/gpl-2.0.txt

//ELF loader from fail0verflow's anergistic.

#include <stdio.h>
#include <string.h>
#include <malloc.h>

#include "config.h"
#include "types.h"
#include "common.h"
#include "spud.h"
#include "disasm.h"
#include "subroutine.h"
#include "block.h"
#include "cfg.h"
#include "registers.h"
#include "graph.h"
#include "output.h"

static void _elf_load_phdr(ctxt_t *ctxt, FILE *fp, u32 phdr_offset, u32 i)
{
	u8 phdr[0x20];
	u32 offset, paddr, size;

	fseek(fp, phdr_offset + 0x20 * i, SEEK_SET);
	fread(phdr, sizeof phdr, 1, fp);

	if(be32(phdr) != 1)
	{
		DBGPRINTF("elf: phdr #%u: no LOAD\n", i);
		return;
	}

	offset = be32(phdr + 0x04);
	paddr = be32(phdr + 0x0c);
	size = be32(phdr + 0x10);
	DBGPRINTF("elf: phdr #%u: 0x%08x bytes (0x%08x -> 0x%08x)\n", i, size, offset, paddr);

	//XXX: integer overflow
	if(offset > LS_SIZE || (offset + size) > LS_SIZE)
		fail("elf: phdr exceeds local storage");

	fseek(fp, offset, SEEK_SET);
	fread(ctxt->ls + paddr, size, 1, fp);
}

static void _elf_extract_execr(ctxt_t *ctxt, FILE *fp, u32 shdr_offset, u32 i)
{
	u8 shdr[0x18];
	u32 addr, size, flags;

	fseek(fp, shdr_offset + 0x28 * i, SEEK_SET);
	fread(shdr, sizeof shdr, 1, fp);

	addr = be32(shdr + 0x0c);
	size = be32(shdr + 0x14);
	flags = be32(shdr + 0x08);
	DBGPRINTF("elf: shdr #%u: 0x%08x bytes (0x%08x)\n", i, size, addr);

	if(flags & 0x04)
	{
		execr_t *er = new execr_t;
		er->ctxt = ctxt;
		er->start = addr;
		er->size = size;
		ctxt->execrs.push_back(er);
		DBGPRINTF("spud: added executable range @ 0x%08x (0x%08x bytes)\n", er->start, er->size);
	}
}

static void _elf_load(ctxt_t *ctxt, const char *path)
{
	static const char elf_magic[] = {0x7f, 'E', 'L', 'F'};
	u8 ehdr[0x34];
	u32 i, phdr_offset, n_phdrs, shdr_offset, n_shdrs;;

	FILE *fp = fopen(path, "rb");
	if(fp == NULL)
		fail("elf: unable to load elf");

	fread(ehdr, sizeof ehdr, 1, fp);
	if(memcmp(ehdr, elf_magic, 4))
		fail("elf: not a ELF file");

	//Get program header infos.
	phdr_offset = be32(ehdr + 0x1c);
	n_phdrs = be16(ehdr + 0x2c);
	//Get section header infos.
	shdr_offset = be32(ehdr + 0x20);
	n_shdrs = be16(ehdr + 0x30);

	DBGPRINTF("elf: %u phdrs at offset 0x%08x\n", n_phdrs, phdr_offset);
	DBGPRINTF("elf: %u shdrs at offset 0x%08x\n", n_shdrs, shdr_offset);

	//Load program headers.
	for(i = 0; i < n_phdrs; i++)
		_elf_load_phdr(ctxt, fp, phdr_offset, i);

	//Extract executable regions.
	for(i = 0; i < n_shdrs; i++)
		_elf_extract_execr(ctxt, fp, shdr_offset, i);

	ctxt->entry = be32(ehdr + 0x18);
	DBGPRINTF("elf: entry is at 0x%08x\n", ctxt->entry);

	fclose(fp);
}

ctxt_t *spud_create_ctxt(const char *elf)
{
	ctxt_t *res = new ctxt_t;

	if((res->ls = (u8 *)malloc(LS_SIZE)) == NULL)
	{
		delete res;
		fail("unable to allocate local storage");
	}

	memset(res->ls, 0, LS_SIZE);

	_elf_load(res, elf);

	return res;
}

void spud_destroy_ctxt(ctxt_t *ctxt)
{
	unsigned int i, j;
	list<subroutine_t *>::iterator it;

	if(ctxt->ls != NULL)
		free(ctxt->ls);

	for(i = 0; i < ctxt->execrs.size(); i++)
		delete ctxt->execrs[i];

	for(it = ctxt->subroutines.begin(); it != ctxt->subroutines.end(); ++it)
	{
		subroutine_t *sr = *it;

		for(j = 0; j < sr->tsubrefs->size(); j++)
			delete (*sr->tsubrefs)[j];
		
		for(j = 0; j < sr->fsubrefs->size(); j++)
			delete (*sr->fsubrefs)[j];

		//FIXME
		//list<block_t *>::iterator it;
		//for(it = sr->blocks.begin(); it != sr->blocks.end(); i++)
			//delete (*it);

		for(j = 0; j < sr->edges->size(); j++)
			delete (*sr->edges)[j];

		delete sr->fsubrefs;
		delete sr->tsubrefs;
		delete sr->edges;
		delete sr->blocks;

		delete sr;
	}

	//TODO: delete other structures.

	delete ctxt;
}

void spud_decompile(ctxt_t *ctxt)
{
	unsigned int i;
	list<subroutine_t *>::iterator it;

	//Disassemble executable regions.
	disasm_disassemble(ctxt);

	//Extract all subroutines.
	subroutine_extract_all(ctxt);
#ifdef _DEBUG
	DBGPRINTF("press any key to continue...");
	getchar();
#endif
	
	//Extract all blocks.
	for(it = ctxt->subroutines.begin(); it != ctxt->subroutines.end(); ++it)
		block_extract_all(*it);
#ifdef _DEBUG
	DBGPRINTF("press any key to continue...");
	getchar();
#endif

	//Build control flow graphs.
	cfg_build_all(ctxt);

	//Analyze register usage.
	registers_analyze_all(ctxt);

	//Traverse control flow graphs.
	for(it = ctxt->subroutines.begin(); it != ctxt->subroutines.end(); ++it)
	{
		DBGPRINTF("traversing sub @ 0x%08x\n", SUBSADDR(*it));
		cfs_traverse(*it, false);

		//Print DFS nodes.
		printf("DFS:\n");
		list<block_t *>::iterator bit;
		for(bit = (*it)->dfsblocks->begin(); bit != (*it)->dfsblocks->end(); ++bit)
		{
			block_t *b = *bit;
			printf("  block @ 0x%08x (%d)\n", BLOCKSADDR(b), b->node.dfsnum);
		}

		//Print dominators.
		printf("dominators:\n");
		for(bit = (*it)->blocks->begin(); bit != (*it)->blocks->end(); ++bit)
		{
			block_t *b = *bit;
			printf("  block @ 0x%08x has dominator @ 0x%08x\n", BLOCKSADDR(b), BLOCKSADDR(b->node.dominator->block));
		}
	}

	for(it = ctxt->subroutines.begin(); it != ctxt->subroutines.end(); ++it)
	{
		DBGPRINTF("reverse traversing sub @ 0x%08x\n", SUBSADDR(*it));
		cfs_traverse(*it, true);

		//Print reverse DFS nodes.
		printf("reverse DFS:\n");
		list<block_t *>::iterator bit;
		for(bit = (*it)->revdfsblocks->begin(); bit != (*it)->revdfsblocks->end(); ++bit)
		{
			block_t *b = *bit;
			printf("  block @ 0x%08x (%d)\n", BLOCKSADDR(b), b->node.dfsnum);
		}

		//Print post dominators.
		printf("post dominators:\n");
		for(bit = (*it)->blocks->begin(); bit != (*it)->blocks->end(); ++bit)
		{
			block_t *b = *bit;
			printf("  block @ 0x%08x has dominator @ 0x%08x\n", BLOCKSADDR(b), BLOCKSADDR(b->revnode.dominator->block));
		}
	}
}
