// Copyright 2011 naehrwert
// Licensed under the terms of the GNU GPL, version 2
// http://www.gnu.org/licenses/old-licenses/gpl-2.0.txt

#include <stdio.h>

#include "config.h"
#include "common.h"
#include "spud.h"
#include "disasm.h"
#include "subroutine.h"
#include "block.h"
#include "util.h"
#include "output.h"

static void output_write_block(FILE *fp, block_t *bl)
{
	unsigned int i;

	//Write header.
	fprintf(fp, "\t//block @ %05x\n\t{\n", IIDX2ADDR(bl->sr->er, bl->sidx));

	//Write instructions for now.
	for(i = bl->sidx; i <= bl->eidx; i++)
	{
		instr_t *inst = &(bl->sr->er->instrs[i]);

		//Check for direct function call.
		if((inst->instr == INSTR_BR || 
			inst->instr == INSTR_BRA || 
			inst->instr == INSTR_BRSL) && 
			subroutine_find_tsubref(bl->sr, BRANCH_TARGET(inst)) != NULL)
			fprintf(fp, "\t\tsub_%05x();\n", BRANCH_TARGET(inst));
		else
		{
			//Fallback: output asm statement.
			fprintf(fp, "\t\tasm(\"");
			disasm_print_instr(fp, inst, false);
			fprintf(fp, "\");\n");
		}
	}

	//Write footer.
	fprintf(fp, "\t}\n");
}

static void output_write_subroutine(FILE *fp, subroutine_t *sr)
{
	u32 i;
	
	//Write used registers.
	fprintf(fp, "//Used registers: ");
	for(i = 0; i < REG_COUNT; i++)
		if(bmap_test_bit(sr->used_regs, i))
			fprintf(fp, "r%d ", i);
	fprintf(fp, "\n");

	//Write if the sub is reachable.
	if(sr->reachable == false)
		fprintf(fp, "//Seems to be not reachable.\n");
	
	//Write the sub name.
	if(SUBSADDR(sr) == sr->er->ctxt->entry)
		fprintf(fp, "void _start() ");
	else
		fprintf(fp, "sub_%05x() ", IIDX2ADDR(sr->er, sr->sidx));

	//The __attribute__((noreturn)) should be added to the prototype and not the function itself.
	if(sr->noreturn == true)
		fprintf(fp, "__attribute__((noreturn)) ");

	fprintf(fp, "\n{");

	//Write all blocks.
	list<block_t *>::iterator iter;
	for(iter = sr->blocks->begin(); iter != sr->blocks->end() && (*iter)->type != BLOCK_END; ++iter)
	{
		block_t *bl = *iter;

		//Ignore start and end blocks.
		if(bl->type == BLOCK_START || bl->type == BLOCK_END)
			continue;

		//Write block.
		output_write_block(fp, bl);
	}
	
	//Write footer.
	fprintf(fp, "}\n");
}

void output_write_code(ctxt_t *ctxt, const char *file)
{
	unsigned int i;
	list<subroutine_t *>::iterator it;
	FILE *fp = fopen(file, "w");

	if((fp = fopen(file, "w")) == NULL)
		fail("output: could not open file %s", file);

	fprintf(fp, "//This file was generated with spud (c) 2011 by naehrwert\n\n");

	//Write out all subroutines.
	for(it = ctxt->subroutines.begin(); it != ctxt->subroutines.end(); ++it)
	{
		output_write_subroutine(fp, *it);
		fprintf(fp, "\n");
	}

	fclose(fp);
}

//Edge type names.
static const char *et[] = 
{
	"BRANCH",
	"BRANCH_COND",
	"EXIT",
	"NEXT",
	"ERROR"
};

//Block type names.
static const char *bt[] =
{
	"START",
	"SIMPLE",
	"CALL",
	"END"
};

//Block type colors.
static const char *bc[] =
{
	"green",
	"black",
	"blue",
	"red"
};

void _output_write_graph_block(FILE *fp, block_t *bl)
{
	switch(bl->type)
	{
	case BLOCK_START:
		fprintf(fp, "n_start_%05x", SUBSADDR(bl->sr));
		break;
	case BLOCK_END:
		fprintf(fp, "n_end_%05x", SUBEADDR(bl->sr));
		break;
	default:
		fprintf(fp, "n_%05x", BLOCKSADDR(bl));
		break;
	}
}

void _output_write_graph_baddr(FILE *fp, block_t *bl)
{
	switch(bl->type)
	{
	case BLOCK_START:
		fprintf(fp, "%05x", SUBSADDR(bl->sr));
		break;
	case BLOCK_END:
		fprintf(fp, "%05x", SUBEADDR(bl->sr));
		break;
	default:
		fprintf(fp, "%05x", BLOCKSADDR(bl));
		break;
	}
}

void output_write_graphviz(ctxt_t *ctxt, const char *file, u32 flags)
{
	unsigned int i, j;
	list<subroutine_t *>::iterator it;
	FILE *fp = fopen(file, "w");

	if((fp = fopen(file, "w")) == NULL)
		fail("output: could not open file %s", file);

	fprintf(fp, "digraph {\n");
	for(it = ctxt->subroutines.begin(); it != ctxt->subroutines.end(); ++it)
	{
		subroutine_t *sr = *it;

		//Write header.
		fprintf(fp, "\t//subroutine 0x%05x\n", SUBSADDR(sr));
		
		//Write blocks.
		list<block_t *>::iterator it;
		for(it = sr->blocks->begin(); it != sr->blocks->end(); ++it)
		{
			block_t *bl = *it;

			fprintf(fp, "\t");
			_output_write_graph_block(fp, bl);
			fprintf(fp, " [label=\"");
			_output_write_graph_baddr(fp, bl);
			fprintf(fp, " (%s)\" color=\"%s\"]\n", bt[bl->type], bc[bl->type]);

			//Check for call block.
			if(bl->type == BLOCK_CALL)
			{
				//Calls goes to subroutine start.
				fprintf(fp, "\t");
				_output_write_graph_block(fp, bl);
				fprintf(fp, " -> n_start_%05x [label=\"CALL\" color=\"blue\"]\n", (u32)BRANCH_TARGET(bl->exitinst));
			}

			//Write dominator.
			if(flags & OUTGRAPH_WRITE_DOM)
			{
				fprintf(fp, "\td_");
				_output_write_graph_baddr(fp, bl);
				fprintf(fp, " -> d_");
				_output_write_graph_baddr(fp, bl->node.dominator->block);
				fprintf(fp, " [label=\"DOM\"color=\"red\"]\n");
			}
			//Write reverse dominator.
			if(flags & OUTGRAPH_WRITE_RDOM)
			{
				fprintf(fp, "\td_");
				_output_write_graph_baddr(fp, bl);
				fprintf(fp, " -> d_");
				_output_write_graph_baddr(fp, bl->revnode.dominator->block);
				fprintf(fp, " [label=\"RDOM\"color=\"red\"]\n");
			}
		}

		//Write edges.
		for(j = 0; j < sr->edges->size(); j++)
		{
			edge_t *e = (*sr->edges)[j];
			fprintf(fp, "\t");
			_output_write_graph_block(fp, e->from);
			fprintf(fp, " -> ");
			_output_write_graph_block(fp, e->to);
			fprintf(fp, " [label=\"%s\"]\n", et[e->type]);
		}
	}
	fprintf(fp, "}\n");

	fclose(fp);
}
