// Copyright 2011 naehrwert
// Licensed under the terms of the GNU GPL, version 2
// http://www.gnu.org/licenses/old-licenses/gpl-2.0.txt

//Based on Humberto Naves' pspdecompiler.

#include "common.h"
#include "subroutine.h"
#include "block.h"
#include "spud.h"

static void _cfg_dfs_step(block_t *block, bool reverse)
{
	block_t *next;
	block_node_t *node, *nextnode;
	vector<edge_t *> *refs;
	vector<edge_t *>::iterator it;
	list <block_t *> *out;

	if(reverse == true)
	{
		node = &block->revnode;
		refs = block->inedge;
		out = block->sr->revdfsblocks;
	}
	else
	{
		node = &block->node;
		refs = block->outedge;
		out = block->sr->dfsblocks;
	}

	node->dfsnum = -1;

	for(it = refs->begin(); it != refs->end(); ++it)
	{
		edge_t *e = *it;

		if(reverse == true)
		{
			next = e->from;
			nextnode = &next->revnode;
		}
		else
		{
			next = e->to;
			nextnode = &next->node;
		}

		if(nextnode->dfsnum == 0)
		{
			nextnode->parent = node;
			node->children->push_back(nextnode);
			_cfg_dfs_step(next, reverse);
		}
	}

	node->dfsnum = block->sr->tmp--;
	node->block = block;
	out->push_front(block);
}

static bool _cfg_dfs(subroutine_t *sr, bool reverse)
{
	block_t *start = (reverse == true ? sr->blocks->back() : sr->blocks->front());
	
	sr->tmp = sr->blocks->size();
	_cfg_dfs_step(start, reverse);

	if(sr->tmp == 0)
		return true;
	return false;
}

static void _dom_dfs_step(block_node_t *node, int *dom_dfsnum_first, int *dom_dfsnum_last)
{
	block_node_t *next;
	list<block_node_t *>::iterator it;

	node->dom_dfsnum_first = (*dom_dfsnum_first)++;
	for(it = node->domchildren->begin(); it != node->domchildren->end(); ++it)
	{
		next = *it;
		if(next->dom_dfsnum_first == 0)
			_dom_dfs_step(next, dom_dfsnum_first, dom_dfsnum_last);
	}

	node->dom_dfsnum_last = (*dom_dfsnum_last)++;
}

static block_node_t *_dom_common(block_node_t *n1, block_node_t *n2)
{
	while(n1 != n2)
	{
		while(n1->dfsnum > n2->dfsnum)
			n1 = n1->dominator;
		while(n2->dfsnum > n1->dfsnum)
			n2 = n2->dominator;
	}

	return n1;
}

static void _cfg_dominance(subroutine_t *sr, bool reverse)
{
	block_t *start;
	block_node_t *startnode;
	int dom_dfsnum_first, dom_dfsnum_last;
	bool changed = true;
	list<block_t *> *blocks;
	list<block_t *>::iterator bit;
	vector<edge_t *> *refs;
	vector<edge_t *>::iterator eit;

	if(reverse == true)
	{
		blocks = sr->revdfsblocks;
		start = sr->blocks->back();
		startnode = start->revnode.dominator = &start->revnode;
	}
	else
	{
		blocks = sr->dfsblocks;
		start = sr->blocks->front();
		startnode = start->node.dominator = &start->node;
	}

	while(changed == true)
	{
		changed = false;
		bit = blocks->begin();
		++bit;
		for(; bit != blocks->end(); ++bit)
		{
			block_t *block = *bit;
			block_node_t *node, *dom = NULL;
			refs = (reverse == true ? block->outedge : block->inedge);
			for(eit = refs->begin(); eit != refs->end(); ++eit)
			{
				block_t *bref;
				block_node_t *brefnode;
				edge_t *edge;

				edge = *eit;

				if(reverse == true)
				{
					bref = edge->to;
					brefnode = &bref->revnode;
				}
				else
				{
					bref = edge->from;
					brefnode = &bref->node;
				}

				if(brefnode->dominator != NULL)
				{
					if(dom == NULL)
						dom = brefnode;
					else
						dom = _dom_common(dom, brefnode);
				}
			}

			node = (reverse == true ? &block->revnode : &block->node);
			if(dom != node->dominator)
			{
				node->dominator = dom;
				changed = true;
			}
		}
	}

	for(bit = blocks->begin(); bit != blocks->end(); ++bit)
	{
		block_t *block = *bit;
		block_node_t *node;

		node = (reverse == true ? &block->revnode : &block->node);
		node->dominator->domchildren->push_back(node);
	}

	dom_dfsnum_first = 0;
	dom_dfsnum_last = blocks->size();
	_dom_dfs_step(startnode, &dom_dfsnum_first, &dom_dfsnum_last);
}

static void _cfg_frontier(subroutine_t *sr, bool reverse)
{
	block_t *block;
	block_node_t *blocknode, *runner;
	list<block_t *> *blocks;
	list<block_t *>::iterator bit;
	vector<edge_t *> *refs;
	vector<edge_t *>::iterator eit;

	blocks = (reverse == true ? sr->revdfsblocks : sr->dfsblocks);
	for(bit = blocks->begin(); bit != blocks->end(); ++bit)
	{
		block = *bit;

		if(reverse == true)
		{
			refs = block->outedge;
			blocknode = &block->revnode;
		}
		else
		{
			refs = block->inedge;
			blocknode = &block->node;
		}

		if(refs->size() >= 2)
		{
			for(eit = refs->begin(); eit != refs->end(); ++eit)
			{
				edge_t *edge = *eit;
				runner = (reverse == true ? &edge->to->revnode : &edge->from->node);
				while(runner != blocknode->dominator)
				{
					runner->frontier->push_back(blocknode);
					runner = runner->dominator;
				}
			}
		}
	}
}

void cfs_traverse(subroutine_t *sr, bool reverse)
{
	if(reverse == true)
	{
		if(_cfg_dfs(sr, true) == false)
			DBGPRINTF("graph: infinite loop in subroutine @ 0x%08x\n", SUBSADDR(sr));
	}
	else
	{
		if(_cfg_dfs(sr, false) == false)
			DBGPRINTF("graph: unreachable code in subroutine @ 0x%08x\n", SUBSADDR(sr));
	}

	//Find the dominators.
	_cfg_dominance(sr, reverse);
	//Find the frontiers.
	_cfg_frontier(sr, reverse);
}
