// Copyright 2011 naehrwert
// Licensed under the terms of the GNU GPL, version 2
// http://www.gnu.org/licenses/old-licenses/gpl-2.0.txt

#include <stdlib.h>
#include <string.h>

#include "common.h"
#include "config.h"
#include "spud.h"
#include "disasm.h"
#include "subroutine.h"
#include "block.h"

static block_t *_block_copy(block_t *bl)
{
	block_t *res = new block_t;
	memcpy(res, bl, sizeof(block_t));
	return res;
}

static void _block_split(subroutine_t *sr, u32 target, list<block_t *>::iterator &it)
{
	block_t *tbl = *it;

	//Create two new blocks.
	block_t *first = _block_copy(tbl);
	block_t *second = _block_copy(tbl);
	
	//First block ends with the instruction previous the target one.
	first->exitinst = disasm_get_instr(sr->er, target - INSTR_SIZE);
	first->eidx = first->exitinst->idx;
	
	//Second block starts with target instruction.
	second->sidx = disasm_get_instr(sr->er, target)->idx;

	//Replace the blocks in the original list.
	sr->blocks->insert(it, first);
	sr->blocks->insert(it, second);
	sr->blocks->erase(it);

	//Check for call blocks.
	if(tbl->type == BLOCK_CALL)
		first->type = BLOCK_SIMPLE;

	DBGPRINTF("block: split block @ 0x%05x into block @ 0x%05x and block @ 0x%05x\n",
		BLOCKSADDR(tbl), BLOCKSADDR(first), BLOCKSADDR(second));

	//Delete the splitted block.
	delete tbl;
}

static void _block_split_blocks(subroutine_t *sr)
{
	//Outer and inner iterators.
	list<block_t *>::iterator oit, iit;

	for(oit = sr->blocks->begin(); oit != sr->blocks->end() && (*oit)->type != BLOCK_END; ++oit)
	{
		block_t *bl = *oit;

		//Ignore start and end blocks.
		if(bl->type == BLOCK_START || bl->type == BLOCK_END)
			continue;

		if(disasm_is_direct_branch(bl->exitinst))
		{
			u32 target = BRANCH_TARGET(bl->exitinst);

			//Check for a branch target inside another block.
			for(iit = sr->blocks->begin(); iit != sr->blocks->end() && (*iit)->type != BLOCK_END; ++iit)
			{
				block_t *tbl = *iit;

				//Don't check if the target is the first instruction.
				if(target > BLOCKSADDR(tbl) && target <= BLOCKEADDR(tbl))
				{
					if(bl != tbl)
					{
						_block_split(sr, target, iit);
						break;
					}
					else
					{
						block_t *second = _block_copy(tbl);
						//Adjust current block.
						tbl->exitinst = disasm_get_instr(sr->er, target - INSTR_SIZE);
						tbl->eidx = tbl->exitinst->idx;
						//Check for call blocks.
						if(tbl->type == BLOCK_CALL)
							tbl->type = BLOCK_SIMPLE;
						//Second block starts with target instruction.
						second->sidx = disasm_get_instr(sr->er, target)->idx;
						//Insert after current block.
						++iit;
						sr->blocks->insert(iit, second);
						DBGPRINTF("block: split block @ 0x%05x into block @ 0x%05x and block @ 0x%05x (split self)\n",
							BLOCKSADDR(tbl), BLOCKSADDR(tbl), BLOCKSADDR(second));
						break;
					}
				}
			}
		}
	}
}

static void _block_init(subroutine_t *sr)
{
	list<block_t *>::iterator it;

	for(it = sr->blocks->begin(); it != sr->blocks->end(); ++it)
	{
		block_t *bl = *it;

		//HACK: sometimes blocks get the CALL type, I really don't know why, this is just a dirty hack.
		//TODO: find real cause and remove this!
		if(!(bl->type == BLOCK_START || bl->type == BLOCK_END) && 
			!IS_DIRECT_BRANCH(bl->exitinst) && bl->type == BLOCK_CALL)
			bl->type = BLOCK_SIMPLE;
		
		bl->inedge = new vector<edge_t *>();
		bl->outedge = new vector<edge_t *>();

		bl->node.children = new list<block_node_t *>();
		bl->node.domchildren = new list<block_node_t *>();
		bl->node.frontier = new list<block_node_t *>();

		bl->revnode.children = new list<block_node_t *>();
		bl->revnode.domchildren = new list<block_node_t *>();
		bl->revnode.frontier = new list<block_node_t *>();

		bl->node.dfsnum = 0;
		bl->node.dom_dfsnum_first = 0;
		bl->node.dom_dfsnum_last = 0;

		bl->revnode.dfsnum = 0;
		bl->revnode.dom_dfsnum_first = 0;
		bl->revnode.dom_dfsnum_last = 0;

		bl->node.dominator = NULL;
		bl->node.parent = NULL;

		bl->revnode.dominator = NULL;
		bl->revnode.parent = NULL;
	}
}

static block_t *_block_extract(subroutine_t *sr, unsigned int sidx)
{
	unsigned int i;

	for(i = sidx; i <= sr->eidx; i++)
	{
		instr_t *inst = &(sr->er->instrs[i]);

		//Check for a direct branch or if we are at the end of the 
		//subroutine because we need at least one block per subroutine.
		if(disasm_is_direct_branch(inst) || i == sr->eidx)
		{
			block_t *res = new block_t;
			res->sr = sr;
			res->sidx = sidx;
			res->eidx = i;
			res->exitinst = inst;
			
			//Check if the branch occurs to a target inside the subroutine.
			if((BRANCH_TARGET(inst) >= SUBSADDR(sr) && 
				BRANCH_TARGET(inst) <= SUBEADDR(sr)))
				res->type = BLOCK_SIMPLE;
			else //Otherwise the branch is a call to another subroutine.
			{
				res->type = BLOCK_CALL;
				//TODO: Check if this subroutine already exists.
			}
			return res;
		}
	}

	return NULL;
}

void block_extract_all(subroutine_t *sr)
{
	unsigned int i;

	DBGPRINTF("block: extracting from sub @ 0x%05x\n", SUBSADDR(sr));

	//Add start node.
	block_t *start = new block_t;
	start->sr = sr;
	start->type = BLOCK_START;
	sr->blocks->push_back(start);

	i = sr->sidx;
	while(i <= sr->eidx)
	{
		block_t *bl = _block_extract(sr, i);
		if(bl != NULL)
		{
			DBGPRINTF("block: found block @ 0x%05x (end @ 0x%05x)\n", BLOCKSADDR(bl), BLOCKEADDR(bl));
			sr->blocks->push_back(bl);
			//Move instruction to block end.
			i = bl->eidx;
		}
		i++;
	}

	//Add end node.
	block_t *end = new block_t;
	end->sr = sr;
	end->type = BLOCK_END;
	sr->blocks->push_back(end);

	//Split all blocks.
	_block_split_blocks(sr);

	//Create all lists.
	_block_init(sr);
}
