#include "types.h"
#include "aes.h"

#include "keys_341.h"

/*! Key bits. */
#define KEY_BITS(size) ((size) * 8)
/*! MMIO register. */
#define MMIO_REG(base, addr) ((base)+(addr)
/*! Align operation. */
#define ALIGN(x, a) (((x) + (a) - 1) & ~((a) - 1))

u32 mmio_87_check_4()
{
	u32 mask1 = 0x7F000000;
	u32 mask2 = 0x04000000;
	return (dma_get_u32(0x24000087000) & mask1) == mask2;
}

u32 mmio_87_check_3()
{
	u32 mask1 = 0x7F000000;
	u32 mask2 = 0x03000000;
	return (dma_get_u32(0x24000087000) & mask1) == mask2;
}

u32 mmio_87_check_2()
{
	u32 mask1 = 0x7F000000;
	u32 mask2 = 0x02000000;
	return (dma_get_u32(0x24000087000) & mask1) == mask2;
}

u32 mmio_87_check_1()
{
	u32 mask1 = 0x7F000000;
	u32 mask2 = 0x01000000;
	return (dma_get_u32(0x24000087000) & mask1) == mask2;
}

//Real implementation in lv1ldr uses ch74 data as seed.
static u8 _rand[0x14];
void init_rand()
{
	u32 i;
	for(i = 0; i < 0x14; i++)
		_rand[i] = (((u8)i) << 8) | ((u8)i);
}

//Real implementation in lv1ldr uses fips186prng.
void gen_rand(u8 *ptr)
{
	u32 i;
	for(i = 0; i < 0x14; i++)
	{
		ptr[i] = _rand[i];
		_rand[i] = (_rand[i]*i) & 0xFF;
	}
}

//This is done in lv1ldr_main.
void _init()
{
	aes_context aes_ctxt;
	u8 idps[0x10], iv[0x10], iso_root_key[0x20], iso_root_iv[0x10];
	
	//SB init.
	sb_init(ctxt, ctxt->0x28);
	
	//Get iso root keyset.
	memcpy(iso_root_key, (u8 *)0, 0x20);
	memcpy(iso_root_iv, (u8 *)0x20, 0x10);
	
	//Generate individuals from EID0 (ABCA...) and 5930... indiv seeds.
	//...
	
	//Get idps from EID0.
	//e.g. 00 00 00 01 00 81 00 01 03 FF FF FF 18 43 C1 4D
	memset(idps, 0xFF, 0x10);
	get_idps_from_eid0(..., idps);
	
	//...
	
	//Encrypt unknown data (3 * 0x20 bytes) with unknown key/iv.
	aes_setkey_enc(&aes_ctxt, unk_key, KEY_BITS(0x20));
	memcpy(iv, unk_iv, 0x10);
	aes_encrypt_cbc(&aes_ctxt, AES_ENCRYPT, 0x20, iv, unk_data_1, unk_data_1);
	
	aes_setkey_enc(&aes_ctxt, unk_key, KEY_BITS(0x20));
	memcpy(iv, unk_iv, 0x10);
	aes_encrypt_cbc(&aes_ctxt, AES_ENCRYPT, 0x20, iv, unk_data_2, unk_data_2);
	
	aes_setkey_enc(&aes_ctxt, unk_key, KEY_BITS(0x20));
	memcpy(iv, unk_iv, 0x10);
	aes_encrypt_cbc(&aes_ctxt, AES_ENCRYPT, 0x20, iv, unk_data_3, unk_data_3);
	
	//...
	
	//Init device.
	init_device_1(ctxt, ctxt->0x18, ctxt->0x20, ctxt->0x28, idps, iso_root_key, iso_root_iv, 
		init_dev_key_1, init_dev_key_2, init_dev_key_3, init_dev_key_4, 
		unk_data_1, unk_data_2, unk_data_3);
}

typedef struct lv1ldr_params
{
	u64 lv1;
	u64 ea;
	u64 io;
	u64 flags
} lv1ldr_params_t;

/*
ea    0x000000000c297000
io    0x0000000010000000
flags 0x4000000200000003
*/

/*
r3: ctxt *, r4: u64 ea, r5: u64 io, r6: u64 flags, r7: u8 *idps, r8: u8 *iso_root_key, r9: u8 *iso_root_iv, 
r10: u8 *init_dev_key_1, r11: u8 *init_dev_key_2, r12: u8 *init_dev_key_3, r13: u8 *init_dev_key_4, 
r14: u8 *unk_data_1, r15: u8 *unk_data_2, r16: u8 *unk_data_3
*/
void init_device_1(ctxt *, u64 ea, u64 io, u64 flags, u8 *idps, u8 *iso_root_key, u8 *iso_root_iv, 
	u8 *init_dev_key_1, u8 *init_dev_key_2, u8 *init_dev_key_3, u8 *init_dev_key_4, 
	u8 *unk_data_1, u8 *unk_data_2, u8 *unk_data_3)
{
	aes_context aes_ctxt;
	u8 iv[0x10];
	
	u8 encdec_k1[0x20], sb_k2[0x20], encdec_k3[0x20], sb_k4[0x20];
	
	//Generate encdec_k1.
	aes_setkey_enc(&aes_ctxt, iso_root_key, KEY_BITS(0x20);
	memcpy(iv, iso_root_iv, 0x10);
	aes_crypt_cbc(&aes_ctxt, AES_ENCRYPT, 0x20, iv, init_dev_key_1, encdec_k1);
	
	//Generate encdec_k3.
	aes_setkey_enc(&aes_ctxt, iso_root_key, KEY_BITS(0x20);
	memcpy(iv, iso_root_iv, 0x10);
	aes_crypt_cbc(&aes_ctxt, AES_ENCRYPT, 0x20, iv, init_dev_key_3, encdec_k3);
	
	//Generate sb_k2.
	aes_setkey_enc(&aes_ctxt, iso_root_key, KEY_BITS(0x20);
	memcpy(iv, iso_root_iv, 0x10);
	aes_crypt_cbc(&aes_ctxt, AES_ENCRYPT, 0x20, iv, init_dev_key_2, sb_k2);
	
	//Set sb_k4 to zeroes.
	memset(sb_k4, 0, 0x20);
	
	u8 ata_enc_flag = 0xE2; //REFTOOL ata_enc_flag.
	//00 01 00 81 00 01 03 FF FF FF 18 43 C1 4D 00 00
	u16 tid = *(u16 *)(idps+2+2);
	if(tid != 0x81) //REFTOOL
	{
		ata_enc_flag = 0xA2; //RETAIL and DEBUG ata_enc_flag.
		if(tid == 0xA0) //SYSDBG
		{
			ata_enc_flag = 0x22; //SYSBG ata_enc_flag.
			memcpy(sb_k2, unk_data_1, 0x20); //Set sb_k2.
			//00 81 00 01 03 FF FF FF 18 43 C1 4D 00 00 00 01
			u16 v = *(u16 *)(idps+4+2) - 1;
			if(v > 4)
			{
				memcpy(encdec_k1, unk_data_2, 0x20); //Set encdec_k1.
				memcpy(encdec_k3, unk_data_3, 0x20); //Set encdec_k3.
			}
		}
	}
	
	if(mmio_87_check_4())
	{
		ata_enc_flag |= 0x11;
		
		if(tid == 0xA0)
			memcpy(sb_k4, unk_data_1, 0x20); //Set sb_k4.
		else
		{
			//Generate k4.
			aes_setkey_enc(&aes_ctxt, iso_root_key, KEY_BITS(0x20);
			memcpy(iv, iso_root_iv, 0x10);
			aes_crypt_cbc(&aes_ctxt, AES_ENCRYPT, 0x20, iv, init_dev_key_4, sb_k4);
		}
	}
	
	//00 01 00 81 00 01 03 FF FF FF 18 43 C1 4D 00 00
	u32 device = *(u32 *)(idps+2) & 0xFFFF;
	debug_printf("init_device_1 flags %08x, device %d, ata_enc_flag %08x\n", flags, device, ata_enc_flag);
	
	//Get 0x10 random bytes.
	u8 rnd[0x10], _rnd[0x14];
	init_rand();
	gen_rand(_rnd);
	memcpy(rnd, _rnd, 0x10);
	
	//Set sb keys depending on mmio register.
	u8 *fix1, *fix2, *magic;
	fix1 = fix1_dx_0;
	fix2 = fix2_dx_0;
	magic = magic_dx_0;
	if(!mmio_87_check_1())
	{
		fix1 = fix1_px_0;
		fix2 = fix2_px_0;
		magic = magic_px_0;
		if(!mmio_87_check_2())
		{
			fix1 = fix1_dx_1;
			fix2 = fix2_dx_1;
			magic = magic_dx_1;
			if(!mmio_87_check_3())
			{
				fix1 = fix1_px_1;
				fix2 = fix2_px_1;
				magic = magic_px_1;
				if(!mmio_87_check_4())
					return 0x26;
			}
		}
	}
	
	u8 mem[...];
	sub_15AA8(mem/*unused*/, 0x24000000000);
	
	//ata object.
	typedef struct _ata_buf
	{
		//0
		u32 ata_enc_flag;
		//4
		u8 k0[0x18];
		//0x1C
		u8 k1[0x18];
		//0x34
		u8 k2[0x18];
		//0x4C
		u8 k3[0x18];
	} ata_buf_t;
	
	//arg_150
	ata_buf_t ata_buf;
	memset(ata_buf, 0, 0x64);
	ata_buf.ata_enc_flag = ata_enc_flag;
	memcpy(ata_buf.k0, sb_k4, 0x18);
	memcpy(ata_buf.k1, sb_k2, 0x18);
	//k2, k3 are all zero
	
	//Keylist node.
	typedef struct _knode
	{
		u8 key[0x18];
		struct _knode *next;
	} knode_t;
	
	//arg_B0
	knode_t kn_1;
	memset(kn_1, 0, 0x1C);
	memcpy(kn_1.key, encdec_k3, 0x18);
	
	//arg_90
	knode_t kn_0;
	memset(kn_0, 0, 0x1C);
	memcpy(kn_0.key, encdec_k1, 0x18);
	kn_0.next = &kn_1;
	
	//encdec object.
	typedef struct _encdec_buf
	{
		u32 encdec_flag;
		knode_t *klist;
	} encdec_buf_t;
	
	//arg_30
	encdec_buf_t encdec_buf;
	memset(&encdec_buf, 0, 0x08);
	encdec_buf.encdec_flag = 6;
	encdec_buf.klist = &kn_0;
	
	//arg_40
	u8 iv[0x10];
	memset(iv, 0, 0x10);
	
	init_device_2(mem, fix1, fix2, magic, 
	          rnd, iv, ea, io, 
			  &ata_buf, &encdec_buf);
}

/*
r3: u8 *mem_ptr, r4: u8 *fix2, r5: u8 *fix2, r6: u8 *magic, 
r7: u8 *rnd, r8: u8 *iv, r9: u64 ea, r10: u64 io, 
r11: ata_buf_t *ata_buf, r12: encdec_buf_t *encdec_buf
*/
s32 init_device_2(u8 *mem_ptr, u8 *fix2, u8 *fix2, u8 *magic, u8 *rnd, u8 *iv, u64 ea, u64 io, ata_buf_t *ata_buf, encdec_buf_t *encdec_buf)
{
	u32 i, res;
	
	debug_printf("ea %08llx, io %08llx \n", ea, io);
	
	u32 ata_flag = ata_buf->ata_enc_flag
	u32 encdec_flag = encdec_buf->encdec_flag;
	debug_printf("ata flag %08x, encdec flag %08x \n", ata_flag, encdec_flag);
	
	//TODO: check this.
	u32 ata_desc_num = get_desc_num(r3, (ata_flag & 0xF0) >> 4);
	u32 encdec_desc_num = get_desc_num(r3, encdec_flag & 0xFFFF);
	
	debug_printf("ata_desc_num %d, encdec_desc_num %d\n", ata_desc_num, encdec_desc_num);
	
	if((ata_desc_num | encdec_desc_num) == 0)
		return 0;
	
	u8 kgen_obj[...];
	//0x3006000 - encdec
	//0x3005000 - encdec
	if((res = kgen_init(kgen_obj /*unused*/, 0x24000000000, 0x3006000, 0x3005000)) != 0)
	{
		debug_printf("[ERROR]: kgen.init fail %08x\n", res);
		return -1;
	}
	
	u8 kgen_buf[...]
	for(i = 0; i != 9; i++)
	{
		if((res = kgen(kgen_obj, 0x24000000000, iv, rnd, fix1, fix2, kgen_buf, 0x3006000, 0x3005000, 1)) != 0)
		{
			debug_printf("[ERROR]: kgen (gen_rand) fail %08x\n", res);
			return -1;
		}
	}
	if((res = kgen(kgen_obj, 0x24000000000, iv, rnd, fix1, fix2, kgen_buf, 0x3006000, 0x3005000, 0)) != 0)
	{
		debug_printf("[ERROR]: kgen fail %08x\n", res);
		return -1;
	}
	
	//TODO: check this!
	typedef struct _unk_obj
	{
		u32 pad0;
		u32 ea;
		u32 io;
		u32 pad1;
	} unk_obj_t;
	
	//arg_B0
	unk_obj_t obj;
	sub_15248(&obj, ea, io);
	
	//TODO: check this!
	u64 desc = ea;
	u32 src = (u32)(desc + 0x20);
	u32 next = ALIGN(src + 0x10, 0x20);
	put_descriptor(&obj, desc, src, 0x10, next, 0x400, 0);
	
	debug_printf("desc %08x, src %08x, next %08x\n", (u32)desc, src, next);
	
	dma_put_u32(desc + 0x20, *((u32 *)(iv+0x00)));
	dma_put_u32(desc + 0x24, *((u32 *)(iv+0x04)));
	dma_put_u32(desc + 0x28, *((u32 *)(iv+0x08)));
	dma_put_u32(desc + 0x2C, *((u32 *)(iv+0x0C)));
	
	r94 = 0;
	
	for(i = 0; i != ata_desc_num; i++)
	{
		u32 flag = (ata_desc_num - 1 == 0 && encdec_desc_num == 0);
		
		desc = (u64)next;
		src = (u32)(desc + 0x20);
		next = ALIGN(src + 0x40, 0x20);
		
		//TODO: check flag usage!
		r94 = (flag ? 0 : desc);
		next = (flag ? 0 : next);
		put_descriptor(&obj, desc, src, 0x40, next, 0, 0x20);
		
		debug_printf("desc %08x, src %08x, next %08x\n", (u32)desc, src, next);
		
		u32 devidx = get_device_index((ata_buf->ata_enc_flag & 0xF0) >> 4);
		u8 *key;
		
		switch(devidx)
		{
		case 0:
			key = ata_buf->k0;
			break;
		case 1:
			key = ata_buf->k1;
			break;
		case 2:
			key = ata_buf->k2;
			break;
		case 3:
			key = ata_buf->k3;
			break;
		default:
			debug_printf( "[ERROR]: unknown ata device index\n");
			return -0x63;
			break;
		}
		
		u8 ekey[0x18];
		memcpy(ekey, key, 0x18);
		
		//TODO: finish...
		debug_printf("ata enable device %d, enc_enable %d\n", desc, ...);
		enable_ata_device(&obj, (u64)src, magic, , , , , , ekey);
	}
}

/*
r3: unk_obj_t *obj, r4: u64 val0, r5: u8 *magic, r6:, r7:, r8:, r9:, r10:, r11: u8 *key
*/
void enable_ata_device(unk_obj_t *obj, u64 val0, u8 *magic, , , , , , u8 *key)
{
}

/*
r3: unk_obj_t *obj, r4: u64 ea, r5: u64 io
*/
void sub_15248(unk_obj_t *obj, u64 ea, u64 io)
{
	//TODO: check this!
	obj->ea = (u32)ea; //<< 4*8;
	obj->io = (u32)io; //<< 4*8;
}

/*
r3: unk_obj_t *obj, r4: u64 desc, r5: u32 src, r6: u32 src_size, r7: u32 next, r8: u32 val3, r9: u32 val4
*/
void put_descriptor(unk_obj_t *obj, u64 desc, u32 src, u32 src_size, u32 next, u32 val3, u32 val4)
{
	u32 io = obj->io;
	u32 ea = obj->ea;
	
	//src = ea + 0x20 for first call, io = 0
	//next = src + 0x10
	// -> src_dunno = 0x80000020
	// -> next_dunno = 0x80000030
	//src = next + 0x20 for second call, io = 0
	//next = src + 0x40
	// -> src_dunno = 0x80000050
	// ->next_dunno = 0x80000090
	//...
	u32 src_dunno = (src - ea + io) | 0x80000000;
	u32 next_dunno = (next - ea + io) | 0x80000000;
	
	u8 buf[0x20];
	*((u32 *)(buf+0x00)) = src_dunno;
	*((u32 *)(buf+0x04)) = src_size;
	*((u32 *)(buf+0x08)) = 0;
	*((u32 *)(buf+0x0C)) = val4;
	
	//TODO: check this!
	//0 1 2 3 4 5 6 7 8 9 A B C D E F
	*((u32 *)(buf+0x10)) = (next == 0 ? 0 : next_dunno);
	*((u16 *)(buf+0x18)) = 0x4000;
	*((u16 *)(buf+0x1A)) = val3;
	
	dma_memcpy_out(desc, buf, 0x20);
}

/*
r3: u8 *mem_ptr, r4: knode_t *it, r5: u32 idx
*/
//0x15A70
knode_t *get_key192_by_index(u8 *mem_ptr, knode_t *it, u32 idx)
{
	u32 i = 0;
	
	while(it != NULL)
	{
		if(i == idx)
			return it;
		it = it->next;
		i++;
	}
	
	return NULL;
}

/*
r3: u8 *kgen_obj, r4: u64 mmio_base, r5: u8 *iv, r6: u8 *rnd, 
r7: u8 *fix1, r8: u8 *p2, r9: u8 *kgen_buf, 
r10: u64 addr_encdec_1, r11: u64 addr_encdec_2, r12: BOOL gen_rnd
*/
//0x14800
u32 kgen(u8 *kgen_obj, u64 mmio_base, u8 *iv, u8 *rnd, u8 *fix1, u8 *fix2, u8 *kgen_buf, u64 addr_encdec_1, u64 addr_encdec_2, BOOL gen_rnd)
{
	u32 i, v, addr;
	
	//TODO: check this.
	addr = mmio_base + addr_encdec_1;
	v = *((u32 *)(iv+0x0));
	dma_put_u32(addr+0x0, v);
	v = *((u32 *)(iv+0x4));
	dma_put_u32(addr+0x4, v);
	v = *((u32 *)(iv+0x8));
	dma_put_u32(addr+0x8, v);
	v = *((u32 *)(iv+0xC));
	dma_put_u32(addr+0xC, v);
	
	u8 _rnd_buf[0x20];
	u8 *rnd_buf_1 = _rnd_buf; //0xB10
	u8 *rnd_buf_2 = _rnd_buf + 0x10; //0xB20
	for(i = 0; i != 0x18; i+=4) //Copy 0x14 bytes.
		*((u32 *)(rnd_buf_1+i)) *((u32 *)(rnd+i));
	//0 1 2 3 4 5 6 7 8 9 A B C D E F
	*((u32 *)(rnd_buf_2+0x8)) = 0;
	*((u32 *)(rnd_buf_2+0xC)) = 0;
	
	//Encrypt rnd_buf (challenge gen?).
	aes_context aes_ctxt;
	u8 iv[0x10];
	//key = fix1, iv = iv
	aes_setkey_enc(&aes_ctxt, fix1, 0xC0);
	memcpy(iv, iv, 0x10);
	aes_crypt_cbc(&aes_ctxt, AES_ENCRYPT, 0x20, iv, rnd_buf_1, rnd_buf_1);
	
	//TODO: check this.
	//unrolled loop?
	v = *((u32 *)(rand_buf_1+0x0));
	dma_put_u32(addr+0x0, v);
	v = *((u32 *)(rand_buf_1+0x4));
	dma_put_u32(addr+0x4, v);
	v = *((u32 *)(rand_buf_1+0x8));
	dma_put_u32(addr+0x8, v);
	v = *((u32 *)(rand_buf_1+0xC));
	dma_put_u32(addr+0xC, v);
	//
	v = *((u32 *)(rand_buf_2+0x0));
	dma_put_u32(addr+0x10, v);
	v = *((u32 *)(rand_buf_2+0x4));
	dma_put_u32(addr+0x14, v);
	v = *((u32 *)(rand_buf_2+0x8));
	dma_put_u32(addr+0x18, v);
	v = *((u32 *)(rand_buf_2+0xC));
	dma_put_u32(addr+0x1C, v);
	
	//TODO: finish..
}

#define ENCDEC_MMIO_2D90 0x2D90
#define ENCDEC_MMIO_2D98 0x2D98
#define ENCDEC_MMIO_2C10 0x2C10
#define ENCDEC_MMIO_2C14 0x2C14
#define ENCDEC_MMIO_2C18 0x2C18
#define ENCDEC_MMIO_2C1C 0x2C1C
#define ENCDEC_MMIO_2C20 0x2C20
#define ENCDEC_MMIO_2C24 0x2C24
#define ENCDEC_MMIO_2D70 0x2D70
#define ENCDEC_MMIO_2D68 0x2D68
#define ENCDEC_MMIO_2D98 0x2D98
#define ENCDEC_MMIO_2D9C 0x2D9C
//0x24000002C00 - encdec (0x200 bytes)
/*
r3: u8 *kgen_obj, r4: u64 mmio_base, r5: u64 addr_encdec_1, r6: u64 addr_encdec_2
*/
//0x14FE0
u32 kgen_init(u8 *kgen_obj, u64 mmio_base, u64 addr_encdec_1, u64 addr_encdec_2)
{
	u32 v;
	
	v = dma_get_u32(MMIO_REG(mmio_base, ENCDEC_MMIO_2D90));
	if(v & 0x01800000)
	{
		dma_put_u32(MMIO_REG(mmio_base, ENCDEC_MMIO_2D98), 0x20);
		while(dma_get_u32(MMIO_REG(mmio_base, ENCDEC_MMIO_2D98)) != 0x20);
		dma_put_u32(MMIO_REG(mmio_base, ENCDEC_MMIO_2D98), 0x00);
	}
	
	v = dma_get_u32(MMIO_REG(mmio_base, ENCDEC_MMIO_2D90));
	//0xFDFFFFFF = ~0x2000000 = (1<<25) -> Clear bit 25.
	v &= 0xFDFFFFFF;
	dma_put_u32(MMIO_REG(mmio_base, ENCDEC_MMIO_2D90));
	
	dma_put_u32(MMIO_REG(mmio_base, ENCDEC_MMIO_2C10), 0x01);
	dma_put_u32(MMIO_REG(mmio_base, ENCDEC_MMIO_2C14), 0x1F);
	dma_put_u32(MMIO_REG(mmio_base, ENCDEC_MMIO_2C18), 0x1F);
	dma_put_u32(MMIO_REG(mmio_base, ENCDEC_MMIO_2C1C), 0x1F);
	dma_put_u32(MMIO_REG(mmio_base, ENCDEC_MMIO_2C20), 0x1F);
	dma_put_u32(MMIO_REG(mmio_base, ENCDEC_MMIO_2C24), 0x1F);
	
	//TODO: 4*8?
	//03 00 50 00 00 00 00 00
	//00 00 00 00 00 00 00 00
	sub_146F8(MMIO_REG(mmio_base, ENCDEC_MMIO_2D70), addr_encdec_2/* << 4*8*/, 1, 0x1000);
	sub_146F8(MMIO_REG(mmio_base, ENCDEC_MMIO_2D68), addr_encdec_1/* << 4*8*/, 1, 0x1000);
	
	dma_put_u32(MMIO_REG(mmio_base, ENCDEC_MMIO_2D98), 0x30);
	while(dma_get_u32(MMIO_REG(mmio_base, ENCDEC_MMIO_2D98)) != 0x30);
	dma_put_u32(MMIO_REG(mmio_base, ENCDEC_MMIO_2D98), 0x00);
	while(dma_get_u32(MMIO_REG(mmio_base, ENCDEC_MMIO_2D98)) != 0x00);
	
	while((dma_get_u32(MMIO_REG(mmio_base, ENCDEC_MMIO_2D9C)) & 4) != 0x00);
	
	u32 addr = mmio_base + addr_encdec_2 + 0xFF0;
	dma_put_u32(addr, 0x100);
	while(dma_get_u32(addr) != 0x100);
	dma_put_u32(addr, 0x10100);
	while(dma_get_u32(addr) != 0x10100);
	
	return 0;
}

void sub_146F8(r3: u64 mmio_addr, r4: u32, r5: u32, r6: u32)
{
	s32 v;
	
	if((v = get_highest_set_bit(r6)) == -1)
		return;
	
	v -= 1;
	v <<= 3;
	v &= 0x1F8;
	
	r6 = rotqmbyi(r4, -4);
	
	//TODO: wtf??
	//I don't have the nerves to decompile this shit now :D
	
	dma_put_u32(mmio_addr, /*TODO*/);
}

//0x15A08
u32 get_desc_num(r3: unused, r4: u32)
{
	u32 i;
	u32 res = 0;
	
	//TODO: check this.
	for(i = 0; i != 0x20; i++)
	{
		u32 r8 = (r4 & (1 << i)) == 0 ? 0xFFFFFFFF : 0;
		r8 = ~(r8|r8);
		res = r8 - res;
	}
	
	return res;
}

#define MMIO_87020000 0x87020000
#define MMIO_87030000 0x87030000
#define MMIO_31680000 0x31680000
#define MMIO_31600000 0x31600000
void sub_15AA8(r3: u8 *mem_ptr, r4: u64 mmio_base)
{
	u32 v;
	
	v = dma_get_u32(MMIO_REG(mmio_base, MMIO_87020000));
	v |= 4;
	dma_put_u32(MMIO_REG(mmio_base, MMIO_87020000), v);
	
	v = dma_get_u32(MMIO_REG(mmio_base, MMIO_87030000));
	v |= 4;
	dma_put_u32(MMIO_REG(mmio_base, MMIO_87030000), v);
	
	if(!mmio_87_check_100())
		if(mmio_87_check_200())
			return;
	
	sub_14640(MMIO_REG(mmio_base, MMIO_31680000), 0x03000000, 1, 0x00800000);
	sub_14640(MMIO_REG(mmio_base, MMIO_31600000), 0x03800000, 1, 0x00800000);
}

void sub_14640(r3: u64 mmio_addr, r4: u32, r5: u32, r6: u32)
{
	s32 v;
	
	if((v = get_highest_set_bit(r6)) == -1)
		return;
	
	v -= 1;
	v <<= 3;
	v &= 0x1F8;
	
	r6 = rotqmbyi(r4, -4);
	
	//TODO: wtf??
	//I don't have the nerves to decompile this shit now :D
	
	dma_put_u32(mmio_addr, /*TODO*/);
}

//0x14598
s32 get_highest_set_bit(r3: u32)
{
	u32 i;
	
	if(r3 == 0)
		return -1;
	
	for(i = 1; i != 0x20; i++)
		if(spu_rotm(r3, -i) == 0)
			return i - 1;
	
	return -1;
}
