/*-
 * Copyright (C) 2012 glevand <geoffrey.levand@mail.ru>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $FreeBSD$
 */

#include <sys/param.h>
#include <sys/sysctl.h>
#include <sys/lock.h>
#include <sys/mutex.h>
#include <sys/kernel.h>
#include <sys/systm.h>
#include <sys/malloc.h>
#include <sys/module.h>
#include <sys/bus.h>
#include <sys/endian.h>
#include <sys/taskqueue.h>

#include <dev/usb/usb.h>
#include <dev/usb/usbdi.h>
#include "usbdevs.h"

#include <net/ethernet.h>

#include <powerpc/ps3/ps3-hvcall.h>

#include "ps3_eurus.h"
#include "jpt.h"
#include "jptvar.h"

static usb_callback_t jpt_irq_rx_callback;
static usb_callback_t jpt_cmd_tx_callback;
static void jpt_event_task(void *arg, int pending __unused);
static int jpt_event_handler(struct jpt_event_listener *self,
    struct ps3_eurus_event *event);
static void jpt_event_irq(struct jpt_softc *sc, char *buf, int len);
static void jpt_cmd_irq(struct jpt_softc *sc, char *buf, int len);
static int jpt_dev_auth(struct jpt_softc *sc);
static int jpt_dev_init(struct jpt_softc *sc);
static int _jpt_register_event_listener(struct jpt_softc *sc,
    struct jpt_event_listener *listener);
static int _jpt_unregister_event_listener(struct jpt_softc *sc,
    struct jpt_event_listener *listener);
static int _jpt_exec_eurus_cmd(struct jpt_softc *sc,
    enum ps3_eurus_cmd_id cmd, void *payload, unsigned int payload_len,
    unsigned int *resp_status, unsigned int *resp_len, void *resp);

static const STRUCT_USB_HOST_ID jpt_devs[] = {
	{
		USB_VP(USB_VENDOR_SONY, 0x036f),
		USB_IFACE_CLASS(UICLASS_VENDOR),
		USB_IFACE_SUBCLASS(0x2),
		USB_IFACE_PROTOCOL(0x1),
	},
};

static const struct usb_config jpt_usb_config[JPT_N_XFERS] = {
	[JPT_IRQ_RX] = {
		.type = UE_INTERRUPT,
		.endpoint = UE_ADDR_ANY,
		.direction = UE_DIR_IN,
		.bufsize = JPT_IRQ_BUFSZ,
		.flags = {
			.pipe_bof = 1,
			.short_xfer_ok = 1
		},
		.callback = jpt_irq_rx_callback
	},
	[JPT_CMD_TX] = {
		.type = UE_INTERRUPT,
		.endpoint = UE_ADDR_ANY,
		.direction = UE_DIR_OUT,
		.bufsize = JPT_CMD_BUFSZ,
		.flags = {
			.pipe_bof = 1,
			.no_pipe_ok = 1,
		},
		.callback = jpt_cmd_tx_callback
	},
};

static unsigned char jpt_devkey[] = {
	0x76, 0x4e, 0x4b, 0x07, 0x24, 0x42, 0x53, 0xfb,
	0x5a, 0xc7, 0xcc, 0x1d, 0xae, 0x00, 0xc6, 0xd8,
	0x14, 0x40, 0x61, 0x8b, 0x13, 0x17, 0x4d, 0x7c,
	0x3b, 0xb6, 0x90, 0xb8, 0x6e, 0x8b, 0xbb, 0x1d,
};

static device_t jpt_dev;

static MALLOC_DEFINE(M_JPT, "jpt", "PS3 Jupiter driver");

static int
jpt_match(device_t dev)
{
	struct usb_attach_arg *uaa = device_get_ivars(dev);

	if (uaa->usb_mode != USB_MODE_HOST)
		return (ENXIO);

	if (uaa->info.bConfigIndex != JPT_CONFIG_INDEX)
		return (ENXIO);

	if (uaa->info.bIfaceIndex != JPT_IFACE_INDEX)
		return (ENXIO);

	return (usbd_lookup_id_by_uaa(jpt_devs, sizeof(jpt_devs), uaa));
}

static int
jpt_attach(device_t dev)
{
	struct jpt_softc *sc = device_get_softc(dev);
	struct usb_attach_arg *uaa = device_get_ivars(dev);
	uint8_t iface_index = JPT_IFACE_INDEX;
	uint64_t v1, v2;
	usb_error_t uerr;
	int err;

	sc->sc_dev = dev;
	sc->sc_udev = uaa->device;

	device_set_usb_desc(dev);

	mtx_init(&sc->sc_mtx, device_get_nameunit(sc->sc_dev),
	    MTX_NETWORK_LOCK, MTX_DEF);

	uerr = usbd_transfer_setup(uaa->device, &iface_index, sc->sc_xfer,
	    jpt_usb_config, JPT_N_XFERS, sc, &sc->sc_mtx);
	if (uerr) {
		device_printf(dev,
		    "could not allocate USB transfers err=%s\n",
		    usbd_errstr(uerr));
		err = ENXIO;
		goto fail_destroy_mutex;
	}

	mtx_init(&sc->sc_elq_mtx, device_get_nameunit(sc->sc_dev),
	    MTX_NETWORK_LOCK, MTX_DEF);
	TAILQ_INIT(&sc->sc_elq);

	/* register event listener */

	sc->sc_el.el_func = jpt_event_handler;
	sc->sc_el.el_arg = sc;

	err = _jpt_register_event_listener(sc, &sc->sc_el);
	if (err) {
		device_printf(sc->sc_dev,
		    "could not register event listener\n");
		goto fail_unsetup_usb_xfers;
	}

	TAILQ_INIT(&sc->sc_eq);

	TASK_INIT(&sc->sc_event_task, 0, jpt_event_task, sc);

	sc->sc_tq = taskqueue_create("jpt_taskq", M_WAITOK,
	    taskqueue_thread_enqueue, &sc->sc_tq);
	if (!sc->sc_tq) {
		device_printf(sc->sc_dev, "could not create taskqueue\n");
		err = ENXIO;
		goto fail_unregister_event_listener;
	}

	taskqueue_start_threads(&sc->sc_tq, 1, PI_NET, "%s taskq",
	    device_get_nameunit(sc->sc_dev));

	/* authenticate device */

	err = jpt_dev_auth(sc);
	if (err) {
		device_printf(sc->sc_dev, "could not authenticate device\n");
		goto fail_free_taskqueue;
	}

	/* get MAC address */

	err = lv1_net_control(LV1_SB_BUS_ID, LV1_GELIC_DEV_ID,
	    LV1_GET_MAC_ADDRESS, 0, 0, 0, &v1, &v2);
	if (err) {
		device_printf(sc->sc_dev,
		    "could not get MAC address err=%d\n", err);
		err = ENXIO;
		goto fail_free_taskqueue;
	}

	v1 <<= 16;
	memcpy(sc->sc_mac_addr, &v1, ETHER_ADDR_LEN);

	device_printf(sc->sc_dev,
	    "MAC address: %02x:%02x:%02x:%02x:%02x:%02x\n",
	    sc->sc_mac_addr[0], sc->sc_mac_addr[1], sc->sc_mac_addr[2],
	    sc->sc_mac_addr[3], sc->sc_mac_addr[4], sc->sc_mac_addr[5]);

	/* get channel info */

	err = lv1_net_control(LV1_SB_BUS_ID, LV1_GELIC_DEV_ID,
	    LV1_GET_CHANNEL_INFO, 0, 0, 0, &v1, &v2);
	if (err) {
		device_printf(sc->sc_dev,
		    "could not get channel info err=%d\n", err);
	}

	if (err)
		sc->sc_chan_info = (0x7ffull << 48);
	else
		sc->sc_chan_info = v1;

	device_printf(sc->sc_dev, "channel info: %016lx\n", sc->sc_chan_info);

	sc->sc_irq_buf = malloc(JPT_IRQ_BUFSZ, M_JPT, M_NOWAIT);
	if (!sc->sc_irq_buf) {
		err = ENOMEM;
		goto fail_free_taskqueue;
	}

	sc->sc_cmd_buf = malloc(JPT_CMD_BUFSZ, M_JPT, M_NOWAIT);
	if (!sc->sc_cmd_buf) {
		err = ENOMEM;
		goto fail_free_irq_buf;
	}

	mtx_lock(&sc->sc_mtx);
	usbd_transfer_start(sc->sc_xfer[JPT_IRQ_RX]);
	mtx_unlock(&sc->sc_mtx);

	/* initialize device */

	err = jpt_dev_init(sc);
	if (err) {
		device_printf(sc->sc_dev, "could not initialize device\n");
		goto fail_free_cmd_buf;
	}

	jpt_dev = dev;
	sc->sc_dev_ready = 1;

	return (0);

fail_free_cmd_buf:

	free(sc->sc_cmd_buf, M_JPT);

fail_free_irq_buf:

	free(sc->sc_irq_buf, M_JPT);

fail_free_taskqueue:

	taskqueue_free(sc->sc_tq);

fail_unregister_event_listener:

	_jpt_unregister_event_listener(sc, &sc->sc_el);

fail_unsetup_usb_xfers:

	mtx_destroy(&sc->sc_elq_mtx);
	usbd_transfer_unsetup(sc->sc_xfer, JPT_N_XFERS);

fail_destroy_mutex:

	mtx_destroy(&sc->sc_mtx);

	return (err);
}

static int
jpt_detach(device_t dev)
{
	struct jpt_softc *sc = device_get_softc(dev);
	struct jpt_event *event, *event_tmp;

	jpt_dev = NULL;

	if (!device_is_attached(dev))
		return (0);

	sc->sc_dev_ready = 0;

	free(sc->sc_cmd_buf, M_JPT);
	sc->sc_cmd_buf = NULL;
	free(sc->sc_irq_buf, M_JPT);
	sc->sc_irq_buf = NULL;

	taskqueue_drain(sc->sc_tq, &sc->sc_event_task);
	taskqueue_free(sc->sc_tq);

	_jpt_unregister_event_listener(sc, &sc->sc_el);

	TAILQ_FOREACH_SAFE(event, &sc->sc_eq, e_queue, event_tmp) {
		TAILQ_REMOVE(&sc->sc_eq, event, e_queue);
		free(event, M_JPT);
	}

	mtx_destroy(&sc->sc_elq_mtx);
	usbd_transfer_unsetup(sc->sc_xfer, JPT_N_XFERS);
	mtx_destroy(&sc->sc_mtx);

	return (0);
}

static void
jpt_irq_rx_callback(struct usb_xfer *xfer, usb_error_t uerr)
{
	struct jpt_softc *sc = usbd_xfer_softc(xfer);
	struct jpt_pkt_hdr *pkt_hdr;
	struct usb_page_cache *pc;
	int actlen;

	usbd_xfer_status(xfer, &actlen, NULL, NULL, NULL);

	switch (USB_GET_STATE(xfer)) {
	case USB_ST_TRANSFERRED:
		pc = usbd_xfer_get_frame(xfer, 0);
		usbd_copy_out(pc, 0, sc->sc_irq_buf, actlen);

		pkt_hdr = (struct jpt_pkt_hdr *) sc->sc_irq_buf;

		switch (pkt_hdr->type) {
		case JPT_PKT_CMD:
			jpt_cmd_irq(sc, sc->sc_irq_buf, actlen);
		break;
		case JPT_PKT_EVENT:
			jpt_event_irq(sc, sc->sc_irq_buf, actlen);
		break;
		default:
			device_printf(sc->sc_dev,
			    "got unknown IRQ packet type=%d\n",
			    pkt_hdr->type);
		}
	/* FALLTHROUGH */
	case USB_ST_SETUP:
setup:
		usbd_xfer_set_frame_len(xfer, 0, usbd_xfer_max_len(xfer));
		usbd_transfer_submit(xfer);
	break;
	default:
		device_printf(sc->sc_dev, "IRQ failed err=%s\n",
		    usbd_errstr(uerr));

		if (uerr != USB_ERR_CANCELLED) {
			usbd_xfer_set_stall(xfer);
			goto setup;
		}
	}
}

static void
jpt_cmd_tx_callback(struct usb_xfer *xfer, usb_error_t uerr)
{
	struct jpt_softc *sc = usbd_xfer_softc(xfer);
	struct usb_page_cache *pc;

	switch (USB_GET_STATE(xfer)) {
	case USB_ST_SETUP:
		pc = usbd_xfer_get_frame(xfer, 0);
		usbd_copy_in(pc, 0, sc->sc_cmd_buf, sc->sc_cmd_len);
		usbd_xfer_set_frame_len(xfer, 0, sc->sc_cmd_len);
		usbd_xfer_set_priv(xfer, sc);
		usbd_transfer_submit(xfer);
	break;
	case USB_ST_TRANSFERRED:
		usbd_xfer_set_priv(xfer, NULL);
	break;
	default:
		device_printf(sc->sc_dev, "command failed err=%s\n",
		    usbd_errstr(uerr));

		if (uerr != USB_ERR_CANCELLED)
			usbd_xfer_set_stall(xfer);

		sc->sc_cmd_err = uerr;
		wakeup(sc->sc_cmd_buf);
	}
}

static void
jpt_event_task(void *arg, int pending __unused)
{
	struct jpt_softc *sc = arg;
	struct jpt_event *event;
	struct jpt_event_listener *listener;
	int err;

	while (1) {
		mtx_lock(&sc->sc_mtx);

		if (TAILQ_EMPTY(&sc->sc_eq)) {
			mtx_unlock(&sc->sc_mtx);
			break;
		}

		event = TAILQ_FIRST(&sc->sc_eq);
		TAILQ_REMOVE(&sc->sc_eq, event, e_queue);

		mtx_unlock(&sc->sc_mtx);

		/* notify event listeners */

		mtx_lock(&sc->sc_elq_mtx);

		TAILQ_FOREACH(listener, &sc->sc_elq, el_queue) {
			err = listener->el_func(listener,
			    &event->e_eurus_event);
			if (err)
				break;
		}

		mtx_unlock(&sc->sc_elq_mtx);

		free(event, M_JPT);
	}
}

static int
jpt_event_handler(struct jpt_event_listener *self,
    struct ps3_eurus_event *event)
{
	struct jpt_softc *sc = self->el_arg;

	if (event->hdr.type == PS3_EURUS_EVENT_TYPE_0x400) {
		if ((event->hdr.id == 0x8) || (event->hdr.id == 0x10))
			wakeup(&sc->sc_dev_ready);
	}

	return (0);
}

static void
jpt_event_irq(struct jpt_softc *sc, char *buf, int len)
{
	struct jpt_pkt_hdr *pkt_hdr;
	struct jpt_event_hdr *event_hdr;
	struct jpt_event *event;
	int i;

	if (len < sizeof(*pkt_hdr) + sizeof(*event_hdr)) {
		device_printf(sc->sc_dev,
		    "got event IRQ packet with invalid length=%d\n", len);
		return;
	}

	pkt_hdr = (struct jpt_pkt_hdr *) buf;
	event_hdr = (struct jpt_event_hdr *) (pkt_hdr + 1);

	if (len < sizeof(*pkt_hdr) + sizeof(*event_hdr) +
	    event_hdr->count * sizeof(struct ps3_eurus_event)) {
		device_printf(sc->sc_dev,
		    "got event IRQ packet with invalid length=%d\n", len);
		return;
	}

	for (i = 0; i < event_hdr->count; i++) {
		event = malloc(sizeof(*event), M_JPT, M_NOWAIT | M_ZERO);
		if (!event) {
			device_printf(sc->sc_dev,
			    "could not allocate memory for new event\n");
			continue;
		}

		memcpy(&event->e_eurus_event,
		    (char *) event_hdr + sizeof(*event_hdr) +
		    i * sizeof(struct ps3_eurus_event),
		    sizeof(struct ps3_eurus_event));

		event->e_eurus_event.hdr.type =
		    le32dec(&event->e_eurus_event.hdr.type);
		event->e_eurus_event.hdr.id =
		    le32dec(&event->e_eurus_event.hdr.id);
		event->e_eurus_event.hdr.timestamp =
		    le32dec(&event->e_eurus_event.hdr.timestamp);
		event->e_eurus_event.hdr.payload_length =
		    le32dec(&event->e_eurus_event.hdr.payload_length);
		event->e_eurus_event.hdr.unknown =
		    le32dec(&event->e_eurus_event.hdr.unknown);

		TAILQ_INSERT_TAIL(&sc->sc_eq, event, e_queue);
	}

	if (event_hdr->count)
		taskqueue_enqueue(sc->sc_tq, &sc->sc_event_task);
}

static void
jpt_cmd_irq(struct jpt_softc *sc, char *buf, int len)
{
	struct jpt_pkt_hdr *pkt_hdr;
	struct jpt_cmd_hdr *cmd_hdr;
	struct ps3_eurus_cmd_hdr *eurus_cmd_hdr;
	uint16_t cmd_tag, eurus_cmd, eurus_tag, payload_len;

	if (len < sizeof(*pkt_hdr) + sizeof(*cmd_hdr) +
	    sizeof(*eurus_cmd_hdr)) {
		device_printf(sc->sc_dev,
		    "got command IRQ packet with invalid length=%d\n", len);
		return;
	}

	pkt_hdr = (struct jpt_pkt_hdr *) buf;
	cmd_hdr = (struct jpt_cmd_hdr *) (pkt_hdr + 1);
	eurus_cmd_hdr = (struct ps3_eurus_cmd_hdr *) (cmd_hdr + 1);
	payload_len = le16dec(&eurus_cmd_hdr->payload_length);

	if (len < sizeof(*pkt_hdr) + sizeof(*cmd_hdr) +
	    sizeof(*eurus_cmd_hdr) + payload_len) {
		device_printf(sc->sc_dev,
		    "got command IRQ packet with invalid length=%d\n", len);
		return;
	}

	cmd_tag = le16dec(&cmd_hdr->tag);
	if (sc->sc_cmd_tag != cmd_tag)
		device_printf(sc->sc_dev,
		    "got command IRQ packet with invalid command tag, "
		    "got 0x%04x, expected 0x%04x\n", cmd_tag, sc->sc_cmd_tag);

	eurus_cmd = le16dec(&eurus_cmd_hdr->id);
	if ((sc->sc_eurus_cmd + 1) != eurus_cmd)
		device_printf(sc->sc_dev,
		    "got command IRQ packet with invalid EURUS command, "
		    "got 0x%04x, expected 0x%04x\n",
		    eurus_cmd, sc->sc_eurus_cmd);

	eurus_tag = le16dec(&eurus_cmd_hdr->tag);
	if (sc->sc_eurus_tag != eurus_tag)
		device_printf(sc->sc_dev,
		    "got command IRQ packet with invalid EURUS tag, "
		    "got 0x%04x, expected 0x%04x\n",
		    eurus_tag, sc->sc_eurus_tag);

	memcpy(sc->sc_cmd_buf, buf, len);

	sc->sc_cmd_err = 0;
	wakeup(sc->sc_cmd_buf);
}

static int
jpt_dev_auth(struct jpt_softc *sc)
{
	struct usb_device_request req;
	usb_error_t uerr;

	req.bmRequestType = UT_WRITE_VENDOR_DEVICE;
	req.bRequest = 0x1;
	USETW(req.wValue, 0x9);
	USETW(req.wIndex, 0x0);
	USETW(req.wLength, sizeof(jpt_devkey));

	uerr =  usbd_do_request_flags(sc->sc_udev, NULL, &req, jpt_devkey,
	    0, NULL, 5000 /* ms */);
	if (uerr)
		return (ENXIO);

	req.bmRequestType = UT_READ_VENDOR_DEVICE;
	req.bRequest = 0x0;
	USETW(req.wValue, 0x2);
	USETW(req.wIndex, 0x0);
	USETW(req.wLength, sizeof(sc->sc_dev_status));

	uerr =  usbd_do_request_flags(sc->sc_udev, NULL,
	    &req, &sc->sc_dev_status, 0, NULL, 5000 /* ms */);
	if (uerr)
		return (ENXIO);

	device_printf(sc->sc_dev, "device status: 0x%04x\n",
	    sc->sc_dev_status);

	return (0);
}

static int
jpt_dev_init(struct jpt_softc *sc)
{
	struct ps3_eurus_cmd_0x114f *eurus_cmd_0x114f;
	struct ps3_eurus_cmd_0x116f *eurus_cmd_0x116f;
	struct ps3_eurus_cmd_0x115b *eurus_cmd_0x115b;
	const uint8_t bcast_addr[] = { 0xff, 0xff, 0xff, 0xff, 0xff, 0xff };
	uint8_t h;
	uint32_t tmp;
	struct ps3_eurus_cmd_mcast_addr_filter *eurus_cmd_mcast_addr_filter;
	struct ps3_eurus_cmd_0x110d *eurus_cmd_0x110d;
	struct ps3_eurus_cmd_0x1031 *eurus_cmd_0x1031;
	struct ps3_eurus_cmd_set_mac_addr *eurus_cmd_set_mac_addr;
	struct ps3_eurus_cmd_set_antenna *eurus_cmd_set_antenna;
	struct ps3_eurus_cmd_0x110b *eurus_cmd_0x110b;
	struct ps3_eurus_cmd_0x1109 *eurus_cmd_0x1109;
	struct ps3_eurus_cmd_0x207 *eurus_cmd_0x207;
	struct ps3_eurus_cmd_0x203 *eurus_cmd_0x203;
	struct ps3_eurus_cmd_0x105f *eurus_cmd_0x105f;
	struct ps3_eurus_cmd_get_fw_version *eurus_cmd_get_fw_version;
	unsigned char *buf;
	unsigned int status, resp_len;
	int err = 0;

	buf = malloc(JPT_CMD_BUFSZ, M_TEMP, M_NOWAIT);
	if (!buf)
		return (ENOMEM);

	/* state 1 */

	eurus_cmd_0x114f = (struct ps3_eurus_cmd_0x114f *) buf;
	memset(eurus_cmd_0x114f, 0, sizeof(*eurus_cmd_0x114f));

	err = _jpt_exec_eurus_cmd(sc, PS3_EURUS_CMD_0x114f,
	    eurus_cmd_0x114f, sizeof(*eurus_cmd_0x114f),
	    &status, NULL, NULL);
	if (err)
		goto done;

	/* do not check command status here !!! */

	/* state 2 */

	err = _jpt_exec_eurus_cmd(sc, PS3_EURUS_CMD_0x1171, NULL, 0,
	    &status, NULL, NULL);
	if (err)
		goto done;

	if (status != PS3_EURUS_CMD_OK) {
		err = EIO;
		goto done;
	}

	/* state 3 */

	err = tsleep(&sc->sc_dev_ready, 0, "jptevent", hz);
	if (err)
		goto done;

	/* state 4 */

	eurus_cmd_0x116f = (struct ps3_eurus_cmd_0x116f *) buf;
	memset(eurus_cmd_0x116f, 0, sizeof(*eurus_cmd_0x116f));
	le32enc(&eurus_cmd_0x116f->unknown, 0x1);

	err = _jpt_exec_eurus_cmd(sc, PS3_EURUS_CMD_0x116f,
	    eurus_cmd_0x116f, sizeof(*eurus_cmd_0x116f),
	    &status, NULL, NULL);
	if (err)
		goto done;

	if (status != PS3_EURUS_CMD_OK) {
		err = EIO;
		goto done;
	}

	/* state 5 */

	eurus_cmd_0x115b = (struct ps3_eurus_cmd_0x115b *) buf;
	memset(eurus_cmd_0x115b, 0, sizeof(*eurus_cmd_0x115b));
	le16enc(&eurus_cmd_0x115b->unknown1, 0x1);
	le16enc(&eurus_cmd_0x115b->unknown2, 0x0);
	memcpy(eurus_cmd_0x115b->mac_addr, sc->sc_mac_addr,
	    sizeof(sc->sc_mac_addr));

	err = _jpt_exec_eurus_cmd(sc, PS3_EURUS_CMD_0x115b,
	    eurus_cmd_0x115b, sizeof(*eurus_cmd_0x115b),
	    &status, NULL, NULL);
	if (err)
		goto done;

	if (status != PS3_EURUS_CMD_OK) {
		err = EIO;
		goto done;
	}

	/* state 6 */

	h = ps3_eurus_mcast_addr_hash(bcast_addr);

	eurus_cmd_mcast_addr_filter =
	    (struct ps3_eurus_cmd_mcast_addr_filter *) buf;
	memset(eurus_cmd_mcast_addr_filter, 0,
	    sizeof(*eurus_cmd_mcast_addr_filter));
	le32enc(&tmp, PS3_EURUS_MCAST_ADDR_HASH2VAL(h));
	eurus_cmd_mcast_addr_filter->word[PS3_EURUS_MCAST_ADDR_HASH2POS(h)] |= tmp;

	err = _jpt_exec_eurus_cmd(sc, PS3_EURUS_CMD_SET_MCAST_ADDR_FILTER,
	    eurus_cmd_mcast_addr_filter, sizeof(*eurus_cmd_mcast_addr_filter),
	    &status, NULL, NULL);
	if (err)
		goto done;

	if (status != PS3_EURUS_CMD_OK) {
		err = EIO;
		goto done;
	}

	/* state 7 */

	eurus_cmd_0x110d = (struct ps3_eurus_cmd_0x110d *) buf;
	memset(eurus_cmd_0x110d, 0, sizeof(*eurus_cmd_0x110d));
	le32enc(&eurus_cmd_0x110d->unknown1, 0xffffffff);
	le32enc(&eurus_cmd_0x110d->unknown2, 0xffffffff);
	le32enc(&eurus_cmd_0x110d->unknown3, 0xffffffff);
	le32enc(&eurus_cmd_0x110d->unknown4, 0xffffffff);
	le32enc(&eurus_cmd_0x110d->unknown5, 0xffffffff);
	le32enc(&eurus_cmd_0x110d->unknown6, 0xffffffff);
	le32enc(&eurus_cmd_0x110d->unknown7, 0xffffffff);

	err = _jpt_exec_eurus_cmd(sc, PS3_EURUS_CMD_0x110d,
	    eurus_cmd_0x110d, sizeof(*eurus_cmd_0x110d),
	    &status, NULL, NULL);
	if (err)
		goto done;

	if (status != PS3_EURUS_CMD_OK) {
		err = EIO;
		goto done;
	}

	/* state 8 */

	eurus_cmd_0x1031 = (struct ps3_eurus_cmd_0x1031 *) buf;
	memset(eurus_cmd_0x1031, 0, sizeof(*eurus_cmd_0x1031));
	eurus_cmd_0x1031->unknown1 = 0x0;
	eurus_cmd_0x1031->unknown2 = 0x0;

	err = _jpt_exec_eurus_cmd(sc, PS3_EURUS_CMD_0x1031,
	    eurus_cmd_0x1031, sizeof(*eurus_cmd_0x1031),
	    &status, NULL, NULL);
	if (err)
		goto done;

	if (status != PS3_EURUS_CMD_OK) {
		err = EIO;
		goto done;
	}

	/* state 9 */

	eurus_cmd_set_mac_addr = (struct ps3_eurus_cmd_set_mac_addr *) buf;
	memset(eurus_cmd_set_mac_addr, 0,
	    sizeof(*eurus_cmd_set_mac_addr));
	memcpy(eurus_cmd_set_mac_addr->mac_addr,
	    sc->sc_mac_addr, sizeof(sc->sc_mac_addr));

	err = _jpt_exec_eurus_cmd(sc, PS3_EURUS_CMD_SET_MAC_ADDR,
	    eurus_cmd_set_mac_addr, sizeof(*eurus_cmd_set_mac_addr),
	    &status, NULL, NULL);
	if (err)
		goto done;

	if (status != PS3_EURUS_CMD_OK) {
		err = EIO;
		goto done;
	}

	/* state 10 */

	eurus_cmd_set_antenna = (struct ps3_eurus_cmd_set_antenna *) buf;
	memset(eurus_cmd_set_antenna, 0, sizeof(*eurus_cmd_set_antenna));

	if (((sc->sc_chan_info >> 40) & 0xff) == 0x1) {
		eurus_cmd_set_antenna->unknown1 = 0x1;
		eurus_cmd_set_antenna->unknown2 = 0x0;
	} else if (((sc->sc_chan_info >> 40) & 0xff) == 0x2) {
		eurus_cmd_set_antenna->unknown1 = 0x1;
		eurus_cmd_set_antenna->unknown2 = 0x1;
	} else {
		eurus_cmd_set_antenna->unknown1 = 0x2;
		eurus_cmd_set_antenna->unknown2 = 0x2;
	}

	err = _jpt_exec_eurus_cmd(sc, PS3_EURUS_CMD_SET_ANTENNA,
	    eurus_cmd_set_antenna, sizeof(*eurus_cmd_set_antenna),
	    &status, NULL, NULL);
	if (err)
		goto done;

	if (status != PS3_EURUS_CMD_OK) {
		err = EIO;
		goto done;
	}

	/* state 11 */

	eurus_cmd_0x110b = (struct ps3_eurus_cmd_0x110b *) buf;
	memset(eurus_cmd_0x110b, 0, sizeof(*eurus_cmd_0x110b));
	le32enc(&eurus_cmd_0x110b->unknown1, 0x1);
	le32enc(&eurus_cmd_0x110b->unknown2, 0x200000);

	err = _jpt_exec_eurus_cmd(sc, PS3_EURUS_CMD_0x110b,
	    eurus_cmd_0x110b, sizeof(*eurus_cmd_0x110b),
	    &status, NULL, NULL);
	if (err)
		goto done;

	if (status != PS3_EURUS_CMD_OK) {
		err = EIO;
		goto done;
	}

	/* state 12 */

	eurus_cmd_0x1109 = (struct ps3_eurus_cmd_0x1109 *) buf;
	ps3_eurus_make_cmd_0x1109(eurus_cmd_0x1109, 0x1, 0x0,
	    0x2715, 0x9, 0x12);

	err = _jpt_exec_eurus_cmd(sc, PS3_EURUS_CMD_0x1109,
	    eurus_cmd_0x1109, sizeof(*eurus_cmd_0x1109),
	    &status, NULL, NULL);
	if (err)
		goto done;

	if (status != PS3_EURUS_CMD_OK) {
		err = EIO;
		goto done;
	}

	/* state 13 */

	eurus_cmd_0x207 = (struct ps3_eurus_cmd_0x207 *) buf;
	memset(eurus_cmd_0x207, 0, sizeof(*eurus_cmd_0x207));
	le32enc(&eurus_cmd_0x207->unknown, 0x1);

	err = _jpt_exec_eurus_cmd(sc, PS3_EURUS_CMD_0x207,
	    eurus_cmd_0x207, sizeof(*eurus_cmd_0x207),
	    &status, NULL, NULL);
	if (err)
		goto done;

	if (status != PS3_EURUS_CMD_OK) {
		err = EIO;
		goto done;
	}

	/* state 14 */

	eurus_cmd_0x203 = (struct ps3_eurus_cmd_0x203 *) buf;
	memset(eurus_cmd_0x203, 0, sizeof(*eurus_cmd_0x203));
	le32enc(&eurus_cmd_0x203->unknown, 0x1);

	err = _jpt_exec_eurus_cmd(sc, PS3_EURUS_CMD_0x203,
	    eurus_cmd_0x203, sizeof(*eurus_cmd_0x203),
	    &status, NULL, NULL);
	if (err)
		goto done;

	if (status != PS3_EURUS_CMD_OK) {
		err = EIO;
		goto done;
	}

	/* state 15 */

	eurus_cmd_0x105f = (struct ps3_eurus_cmd_0x105f *) buf;
	memset(eurus_cmd_0x105f, 0, sizeof(*eurus_cmd_0x105f));
	le16enc(&eurus_cmd_0x105f->channel_info, sc->sc_chan_info >> 48);
	memcpy(eurus_cmd_0x105f->mac_addr,
	    sc->sc_mac_addr, sizeof(sc->sc_mac_addr));

	if (((sc->sc_chan_info >> 40) & 0xff) == 0x1) {
		eurus_cmd_0x105f->unknown1 = 0x1;
		eurus_cmd_0x105f->unknown2 = 0x0;
	} else if (((sc->sc_chan_info >> 40) & 0xff) == 0x2) {
		eurus_cmd_0x105f->unknown1 = 0x1;
		eurus_cmd_0x105f->unknown2 = 0x1;
	} else {
		eurus_cmd_0x105f->unknown1 = 0x2;
		eurus_cmd_0x105f->unknown2 = 0x2;
	}

	err = _jpt_exec_eurus_cmd(sc, PS3_EURUS_CMD_0x105f,
	    eurus_cmd_0x105f, sizeof(*eurus_cmd_0x105f),
	    &status, NULL, NULL);
	if (err)
		goto done;

	if (status != PS3_EURUS_CMD_OK) {
		err = EIO;
		goto done;
	}

	/* device is ready now */

	/* read firmware version */

	eurus_cmd_get_fw_version = (struct ps3_eurus_cmd_get_fw_version *) buf;
	memset(eurus_cmd_get_fw_version, 0,
	    sizeof(*eurus_cmd_get_fw_version));

	err = _jpt_exec_eurus_cmd(sc, PS3_EURUS_CMD_GET_FW_VERSION,
	    eurus_cmd_get_fw_version, sizeof(*eurus_cmd_get_fw_version),
	    &status, &resp_len, eurus_cmd_get_fw_version);
	if (err)
		goto done;

	if (status != PS3_EURUS_CMD_OK) {
		err = EIO;
		goto done;
	}

	device_printf(sc->sc_dev, "firmware version: %s\n",
	    (char *) eurus_cmd_get_fw_version->version);

done:

	free(buf, M_TEMP);

	return (err);
}

static int
_jpt_register_event_listener(struct jpt_softc *sc,
    struct jpt_event_listener *listener)
{
	mtx_lock(&sc->sc_elq_mtx);
	TAILQ_INSERT_TAIL(&sc->sc_elq, listener, el_queue);
	mtx_unlock(&sc->sc_elq_mtx);

	return (0);
}

static int
_jpt_unregister_event_listener(struct jpt_softc *sc,
    struct jpt_event_listener *listener)
{
	mtx_lock(&sc->sc_elq_mtx);
	TAILQ_REMOVE(&sc->sc_elq, listener, el_queue);
	mtx_unlock(&sc->sc_elq_mtx);

	return (0);
}

static int
_jpt_exec_eurus_cmd(struct jpt_softc *sc, enum ps3_eurus_cmd_id cmd,
    void *payload, unsigned int payload_len, unsigned int *resp_status,
    unsigned int *resp_len, void *resp)
{
	struct jpt_pkt_hdr *pkt_hdr;
	struct jpt_cmd_hdr *cmd_hdr;
	struct ps3_eurus_cmd_hdr *eurus_cmd_hdr;
	struct ps3_eurus_cmd_get_channel_info *eurus_cmd_get_channel_info;
	uint16_t status;
	int err;

	if (!payload && payload_len)
		return (EINVAL);

	mtx_lock(&sc->sc_mtx);

	if (sc->sc_cmd_busy) {
		mtx_unlock(&sc->sc_mtx);
		return (EAGAIN);
	}

	sc->sc_cmd_busy = 1;

	mtx_unlock(&sc->sc_mtx);

	/* internal commands */

	if (cmd == PS3_EURUS_CMD_GET_CHANNEL_INFO) {
		if (payload_len < sizeof(*eurus_cmd_get_channel_info)) {
			err = EINVAL;
			goto done;
		}

		if (resp_status)
			*resp_status = PS3_EURUS_CMD_OK;

		if (resp_len && resp) {
			*resp_len = sizeof(*eurus_cmd_get_channel_info);
			eurus_cmd_get_channel_info =
			    (struct ps3_eurus_cmd_get_channel_info *) resp;
			memset(eurus_cmd_get_channel_info,
			    0, sizeof(*eurus_cmd_get_channel_info));
			eurus_cmd_get_channel_info->channel_info =
			    sc->sc_chan_info >> 48;
		}

		err = 0;

		goto done;
	}

	pkt_hdr = (struct jpt_pkt_hdr *) sc->sc_cmd_buf;
	memset(pkt_hdr, 0, sizeof(*pkt_hdr));
	pkt_hdr->unknown1 = 1;
	pkt_hdr->unknown2 = 1;
	pkt_hdr->type = JPT_PKT_CMD;

	cmd_hdr = (struct jpt_cmd_hdr *) (pkt_hdr + 1);
	memset(cmd_hdr, 0, sizeof(*cmd_hdr));
	sc->sc_cmd_tag++;
	cmd_hdr->unknown1 = 0;
	le16enc(&cmd_hdr->unknown2, 1);
	le16enc(&cmd_hdr->tag, sc->sc_cmd_tag);

	eurus_cmd_hdr = (struct ps3_eurus_cmd_hdr *) (cmd_hdr + 1);
	memset(eurus_cmd_hdr, 0, sizeof(*eurus_cmd_hdr));
	sc->sc_eurus_cmd = cmd;
	le16enc(&eurus_cmd_hdr->id, cmd);
	sc->sc_eurus_tag++;
	le16enc(&eurus_cmd_hdr->tag, sc->sc_eurus_tag);
	le16enc(&eurus_cmd_hdr->status, 0xa);
	le16enc(&eurus_cmd_hdr->payload_length, payload_len);

	if (payload_len)
		memcpy(eurus_cmd_hdr + 1, payload, payload_len);

	sc->sc_cmd_len = sizeof(*pkt_hdr) + sizeof(*cmd_hdr) +
	    sizeof(*eurus_cmd_hdr) + payload_len;

	mtx_lock(&sc->sc_mtx);
	usbd_transfer_start(sc->sc_xfer[JPT_CMD_TX]);
	mtx_unlock(&sc->sc_mtx);

	/* wait until command is finished */

	err = tsleep(sc->sc_cmd_buf, 0, "jptcmd", hz);
	if (err)
		goto done;

	err = sc->sc_cmd_err;
	if (!err) {
		status = le16dec(&eurus_cmd_hdr->status);

		if (resp_status)
			*resp_status = status;

		if (resp_len && resp) {
			*resp_len = le16dec(&eurus_cmd_hdr->payload_length);
			memcpy(resp, eurus_cmd_hdr + 1, *resp_len);
		}

		if (status != PS3_EURUS_CMD_OK) {
			device_printf(sc->sc_dev,
			    "EURUS command 0x%04x status=0x%04x\n",
			    cmd, status);
		}
	}

done:

	if (err) {
		device_printf(sc->sc_dev,
		    "EURUS command 0x%04x failed err=%d\n", cmd, err);
	}

	sc->sc_cmd_busy = 0;

	return (err);
}

int
jpt_register_event_listener(struct jpt_event_listener *listener)
{
	struct jpt_softc *sc;

	if (!jpt_dev)
		return (ENODEV);

	sc = device_get_softc(jpt_dev);

	if (!sc->sc_dev_ready)
		return (ENODEV);

	return (_jpt_register_event_listener(sc, listener));
}

int
jpt_unregister_event_listener(struct jpt_event_listener *listener)
{
	struct jpt_softc *sc;

	if (!jpt_dev)
		return (ENODEV);

	sc = device_get_softc(jpt_dev);

	if (!sc->sc_dev_ready)
		return (ENODEV);

	return (_jpt_unregister_event_listener(sc, listener));
}

int
jpt_exec_eurus_cmd(enum ps3_eurus_cmd_id cmd, void *payload,
    unsigned int payload_len, unsigned int *resp_status,
    unsigned int *resp_len, void *resp)
{
	struct jpt_softc *sc;

	if (!jpt_dev)
		return (ENODEV);

	sc = device_get_softc(jpt_dev);

	if (!sc->sc_dev_ready)
		return (ENODEV);

	return (_jpt_exec_eurus_cmd(sc, cmd, payload, payload_len,
	    resp_status, resp_len, resp));
}

static device_method_t jpt_methods[] = {
	DEVMETHOD(device_probe, jpt_match),
	DEVMETHOD(device_attach, jpt_attach),
	DEVMETHOD(device_detach, jpt_detach),
	{ 0, 0 }
};

static driver_t jpt_driver = {
	.name = "jpt",
	.methods = jpt_methods,
	.size = sizeof(struct jpt_softc)
};

static devclass_t jpt_devclass;

DRIVER_MODULE(jpt, uhub, jpt_driver, jpt_devclass, NULL, 0);
MODULE_DEPEND(jpt, usb, 1, 1, 1);
MODULE_VERSION(jpt, 1);
