/*-
 * Copyright (C) 2012 glevand <geoffrey.levand@mail.ru>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer,
 *    without modification, immediately at the beginning of the file.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $FreeBSD$
 */

#include "opt_inet.h"
#include "opt_wlan.h"

#include <sys/param.h>
#include <sys/sysctl.h>
#include <sys/lock.h>
#include <sys/mutex.h>
#include <sys/kernel.h>
#include <sys/systm.h>
#include <sys/malloc.h>
#include <sys/module.h>
#include <sys/bus.h>
#include <sys/sockio.h>
#include <sys/mbuf.h>
#include <sys/priv.h>
#include <sys/socket.h>
#include <sys/endian.h>

#include <dev/usb/usb.h>
#include <dev/usb/usbdi.h>
#include "usbdevs.h"

#include <net/if.h>
#include <net/if_arp.h>
#include <net/if_dl.h>
#include <net/ethernet.h>
#include <net/if_types.h>
#include <net/if_media.h>
#include <net/route.h>

#include <net80211/ieee80211_var.h>
#include <net80211/ieee80211_ioctl.h>

#include "ps3_eurus.h"
#include "jpt.h"
#include "jstavar.h"

static usb_callback_t jsta_rx_callback;
static usb_callback_t jsta_tx_callback;
static void jsta_setup_media(struct jsta_softc *sc);
static void jsta_init(void *arg);
static void jsta_init_locked(struct jsta_softc *sc);
static void jsta_start(struct ifnet *ifp);
static void jsta_start_locked(struct ifnet *ifp);
static void jsta_stop(struct jsta_softc *sc);
static int jsta_ioctl(struct ifnet *ifp, u_long cmd, caddr_t data);
static int jsta_media_change(struct ifnet *ifp);
static void jsta_media_status(struct ifnet *ifp, struct ifmediareq *imr);
static int jsta_event_handler(struct jpt_event_listener *self,
    struct ps3_eurus_event *event);
static int jsta_scan(struct jsta_softc *sc, struct ieee80211_scan_req *sr);
static void jsta_free_scan_results(struct jsta_softc *sc);
static int jsta_get_scan_results(struct jsta_softc *sc);
static int jsta_ioctl_chaninfo(struct jsta_softc *sc,
    struct ieee80211req *ireq);
static int jsta_ioctl_scan_results(struct jsta_softc *sc,
    struct ieee80211req *ireq);
static int jsta_get_mac_addr(struct jsta_softc *sc);
static int jsta_set_mac_addr(struct jsta_softc *sc,
    const uint8_t mac_addr[ETHER_ADDR_LEN]);
static int jsta_get_chan_info(struct jsta_softc *sc);

static const STRUCT_USB_HOST_ID jsta_devs[] = {
	{
		USB_VP(USB_VENDOR_SONY, 0x036f),
		USB_IFACE_CLASS(UICLASS_VENDOR),
		USB_IFACE_SUBCLASS(0x2),
		USB_IFACE_PROTOCOL(0x2),
	},
};

static const struct usb_config jsta_usb_config[JSTA_N_XFERS] = {
	[JSTA_RX] = {
		.type = UE_BULK,
		.endpoint = UE_ADDR_ANY,
		.direction = UE_DIR_IN,
		.bufsize = JSTA_RX_BUFSZ,
		.flags = {
			.pipe_bof = 1,
			.short_xfer_ok = 1
		},
		.callback = jsta_rx_callback
	},
	[JSTA_TX] = {
		.type = UE_BULK,
		.endpoint = UE_ADDR_ANY,
		.direction = UE_DIR_OUT,
		.bufsize = JSTA_TX_BUFSZ,
		.flags = {
			.pipe_bof = 1,
			.short_xfer_ok = 1
		},
		.callback = jsta_tx_callback
	},
};

static const int jsta_rates_11b[] = {
	2, 4, 11, 22,
};

static const int jsta_rates_11g[] = {
	2, 4, 11, 22, 12, 18, 24, 36, 48, 72, 96, 108,
};

static MALLOC_DEFINE(M_JSTA, "jsta", "PS3 Jupiter STA driver");

static int
jsta_match(device_t dev)
{
	struct usb_attach_arg *uaa = device_get_ivars(dev);

	if (uaa->usb_mode != USB_MODE_HOST)
		return (ENXIO);

	if (uaa->info.bConfigIndex != JSTA_CONFIG_INDEX)
		return (ENXIO);

	if (uaa->info.bIfaceIndex != JSTA_IFACE_INDEX)
		return (ENXIO);

	return (usbd_lookup_id_by_uaa(jsta_devs, sizeof(jsta_devs), uaa));
}

static int
jsta_attach(device_t dev)
{
	struct jsta_softc *sc = device_get_softc(dev);
	struct usb_attach_arg *uaa = device_get_ivars(dev);
	uint8_t iface_index = JSTA_IFACE_INDEX;
	struct ifnet *ifp;
	usb_error_t uerr;
	int err;

	sc->sc_dev = dev;
	sc->sc_udev = uaa->device;

	device_set_usb_desc(dev);

	mtx_init(&sc->sc_mtx, device_get_nameunit(sc->sc_dev),
	    MTX_NETWORK_LOCK, MTX_DEF);

	uerr = usbd_transfer_setup(uaa->device, &iface_index, sc->sc_xfer,
	    jsta_usb_config, JSTA_N_XFERS, sc, &sc->sc_mtx);
	if (uerr) {
		device_printf(dev,
		    "could not allocate USB transfers err=%s\n",
		    usbd_errstr(uerr));
		err = ENXIO;
		goto fail_destroy_mutex;
	}

	TAILQ_INIT(&sc->sc_srq);

	/* register event listener */

	sc->sc_el.el_func = jsta_event_handler;
	sc->sc_el.el_arg = sc;

	err = jpt_register_event_listener(&sc->sc_el);
	if (err) {
		device_printf(sc->sc_dev,
		    "could not register event listener\n");
		goto fail_unsetup_usb_xfers;
	}

	err = jsta_get_mac_addr(sc);
	if (err) {
		device_printf(sc->sc_dev, "could not get MAC address\n");
		goto fail_unregister_event_listener;
	}

	err = jsta_set_mac_addr(sc, sc->sc_mac_addr);
	if (err) {
		device_printf(sc->sc_dev, "could not set MAC address\n");
		goto fail_unregister_event_listener;
	}

	err = jsta_get_chan_info(sc);
	if (err) {
		device_printf(sc->sc_dev, "could not get channel info\n");
		goto fail_unregister_event_listener;
	}

	ifp = sc->sc_ifp = if_alloc(IFT_ETHER);
	if (!ifp) {
		device_printf(sc->sc_dev,
		    "could not allocate ifnet structure\n");
		goto fail_unregister_event_listener;
	}

	ifp->if_softc = sc;
	if_initname(ifp, device_get_name(sc->sc_dev),
	    device_get_unit(sc->sc_dev));
	ifp->if_flags = IFF_BROADCAST | IFF_SIMPLEX | IFF_MULTICAST;
	ifp->if_init = jsta_init;
	ifp->if_start = jsta_start;
	ifp->if_ioctl = jsta_ioctl;
	ifp->if_baudrate = 10000000;
	IFQ_SET_MAXLEN(&ifp->if_snd, ifqmaxlen);
	ifp->if_snd.ifq_drv_maxlen = ifqmaxlen;
	IFQ_SET_READY(&ifp->if_snd);

	jsta_setup_media(sc);

	ether_ifattach(ifp, sc->sc_mac_addr);

	return (0);

fail_unregister_event_listener:

	jpt_unregister_event_listener(&sc->sc_el);

fail_unsetup_usb_xfers:

	usbd_transfer_unsetup(sc->sc_xfer, JSTA_N_XFERS);

fail_destroy_mutex:

	mtx_destroy(&sc->sc_mtx);

	return (err);
}

static int
jsta_detach(device_t dev)
{
	struct jsta_softc *sc = device_get_softc(dev);
	struct ifnet *ifp = sc->sc_ifp;

	if (!device_is_attached(dev))
		return (0);

	mtx_lock(&sc->sc_mtx);
	jsta_stop(sc);
	ifmedia_removeall(&sc->sc_ifmedia);
	ifp->if_drv_flags &= ~IFF_DRV_RUNNING;
	mtx_unlock(&sc->sc_mtx);

	ether_ifdetach(ifp);
	if_free(ifp);

	jpt_unregister_event_listener(&sc->sc_el);

	usbd_transfer_unsetup(sc->sc_xfer, JSTA_N_XFERS);

	mtx_destroy(&sc->sc_mtx);

	return (0);
}

static void
jsta_rx_callback(struct usb_xfer *xfer, usb_error_t uerr)
{
}

static void
jsta_tx_callback(struct usb_xfer *xfer, usb_error_t uerr)
{
}

static void
jsta_setup_media(struct jsta_softc *sc)
{
#define N(a)	(sizeof(a) / sizeof((a)[0]))

	int mword, i;

	ifmedia_init(&sc->sc_ifmedia, 0, jsta_media_change,
	    jsta_media_status);

	ifmedia_add(&sc->sc_ifmedia,
	    IFM_MAKEWORD(IFM_IEEE80211, IFM_AUTO, 0, 0), 0, NULL);
	ifmedia_add(&sc->sc_ifmedia,
	    IFM_MAKEWORD(IFM_IEEE80211, IFM_AUTO, IFM_IEEE80211_ADHOC, 0),
	    0, NULL);

	ifmedia_add(&sc->sc_ifmedia,
	    IFM_MAKEWORD(IFM_IEEE80211, IFM_AUTO, IFM_IEEE80211_11B, 0),
	    0, NULL);

	for (i = 0; i < N(jsta_rates_11b); i++) {
		mword = ieee80211_rate2media(NULL, jsta_rates_11b[i],
		    IEEE80211_MODE_11B);
		if (!mword)
			continue;

		ifmedia_add(&sc->sc_ifmedia,
		    IFM_MAKEWORD(IFM_IEEE80211, mword, 0, 0),
		    0, NULL);
		ifmedia_add(&sc->sc_ifmedia,
		    IFM_MAKEWORD(IFM_IEEE80211, mword, IFM_IEEE80211_ADHOC, 0),
		    0, NULL);
	}

	ifmedia_add(&sc->sc_ifmedia,
	    IFM_MAKEWORD(IFM_IEEE80211, IFM_AUTO, IFM_IEEE80211_11G, 0),
	    0, NULL);

	for (i = 0; i < N(jsta_rates_11g); i++) {
		mword = ieee80211_rate2media(NULL, jsta_rates_11g[i],
		    IEEE80211_MODE_11G);
		if (!mword)
			continue;

		ifmedia_add(&sc->sc_ifmedia,
		    IFM_MAKEWORD(IFM_IEEE80211, mword, 0, 0),
		    0, NULL);
		ifmedia_add(&sc->sc_ifmedia,
		    IFM_MAKEWORD(IFM_IEEE80211, mword, IFM_IEEE80211_ADHOC, 0),
		    0, NULL);
	}

	ifmedia_set(&sc->sc_ifmedia,
	    IFM_MAKEWORD(IFM_IEEE80211, IFM_AUTO, 0, 0));

#undef N
}

static void
jsta_init(void *arg)
{
	struct jsta_softc *sc = arg;

	mtx_lock(&sc->sc_mtx);
	jsta_init_locked(sc);
	mtx_unlock(&sc->sc_mtx);
}

static void
jsta_init_locked(struct jsta_softc *sc)
{
	struct ifnet *ifp = sc->sc_ifp;
	int i;

	mtx_assert(&sc->sc_mtx, MA_OWNED);

	if (ifp->if_drv_flags & IFF_DRV_RUNNING)
		jsta_stop(sc);

	sc->sc_associated = 0;

	bzero(sc->sc_ssid, sizeof(sc->sc_ssid));
	sc->sc_ssid_len = 0;

	bzero(sc->sc_stationname, sizeof(sc->sc_stationname));
	sc->sc_stationname_len = 0;

	/* channel list */

	bzero(sc->sc_chanlist, sizeof(sc->sc_chanlist));

	for (i = 1; i <= 14; i++) {
		if (sc->sc_chan_info & (1 << (i - 1)))
			setbit(sc->sc_chanlist, i);
	}

	sc->sc_scanvalid = IEEE80211_SCAN_VALID_DEFAULT;

	ifp->if_drv_flags |= IFF_DRV_RUNNING;
	ifp->if_drv_flags &= ~IFF_DRV_OACTIVE;
}

static void
jsta_start(struct ifnet *ifp)
{
	struct jsta_softc *sc = ifp->if_softc;

	mtx_lock(&sc->sc_mtx);
	jsta_start_locked(ifp);
	mtx_unlock(&sc->sc_mtx);
}

static void
jsta_start_locked(struct ifnet *ifp)
{
	struct jsta_softc *sc = ifp->if_softc;

	mtx_assert(&sc->sc_mtx, MA_OWNED);

	if (ifp->if_drv_flags & IFF_DRV_OACTIVE)
		return;

	if (!sc->sc_associated)
		return;
}

static void
jsta_stop(struct jsta_softc *sc)
{
	struct ifnet *ifp = sc->sc_ifp;

	mtx_assert(&sc->sc_mtx, MA_OWNED);

	ifp->if_drv_flags &= ~(IFF_DRV_RUNNING | IFF_DRV_OACTIVE);

	jsta_free_scan_results(sc);
}

static int
jsta_ioctl(struct ifnet *ifp, u_long cmd, caddr_t data)
{
	struct jsta_softc *sc = ifp->if_softc;
	struct ifreq *ifr = (struct ifreq *) data;
	struct ieee80211req *ireq = (struct ieee80211req *) data;
	struct ieee80211_scan_req sr;
	char tmpstr[IEEE80211_NWID_LEN];
	uint8_t bssid[IEEE80211_ADDR_LEN];
	static const uint8_t zerobssid[IEEE80211_ADDR_LEN];
	uint8_t key[IEEE80211_KEYBUF_SIZE];
	u_int kid, len;
	int err = 0;

	switch (cmd) {
	case SIOCSIFFLAGS:
		mtx_lock(&sc->sc_mtx);

		if (ifp->if_flags & IFF_UP) {
			jsta_init_locked(sc);
		} else {
			if (ifp->if_drv_flags & IFF_DRV_RUNNING)
				jsta_stop(sc);
		}

		mtx_unlock(&sc->sc_mtx);
	break;
	case SIOCSIFMEDIA:
	case SIOCGIFMEDIA:
		err = ifmedia_ioctl(ifp, ifr, &sc->sc_ifmedia, cmd);
	break;
	case SIOCADDMULTI:
	case SIOCDELMULTI:
		err = 0;
	break;
	case SIOCG80211:
		switch (ireq->i_type) {
		case IEEE80211_IOC_NUMSSIDS:
			ireq->i_val = 1;
		break;
		case IEEE80211_IOC_SSID:
			ireq->i_len = sc->sc_ssid_len;
			err = copyout(sc->sc_ssid, ireq->i_data,
			    sc->sc_ssid_len);
		break;
		case IEEE80211_IOC_STATIONNAME:
			ireq->i_len = sc->sc_stationname_len;
			err = copyout(sc->sc_stationname, ireq->i_data,
			    sc->sc_stationname_len);
		break;
		case IEEE80211_IOC_CHANINFO:
			err = jsta_ioctl_chaninfo(sc, ireq);
		break;
		case IEEE80211_IOC_BSSID:
			if (ireq->i_len != IEEE80211_ADDR_LEN)
				return (EINVAL);

			err = copyout(sc->sc_des_bssid, ireq->i_data,
			    ireq->i_len);
		break;
		case IEEE80211_IOC_AUTHMODE:
			if (sc->sc_iflags & IEEE80211_F_WPA)
				ireq->i_val = IEEE80211_AUTH_WPA;
		break;
		case IEEE80211_IOC_PRIVACY:
			ireq->i_val = (sc->sc_iflags & IEEE80211_F_PRIVACY) != 0;
		break;
		case IEEE80211_IOC_WEP:
			if ((sc->sc_iflags & IEEE80211_F_PRIVACY) == 0)
				ireq->i_val = IEEE80211_WEP_OFF;
			else if (sc->sc_iflags & IEEE80211_F_DROPUNENC)
				ireq->i_val = IEEE80211_WEP_ON;
			else
				ireq->i_val = IEEE80211_WEP_MIXED;
		break;
		case IEEE80211_IOC_NUMWEPKEYS:
			ireq->i_val = IEEE80211_WEP_NKID;
		break;
		case IEEE80211_IOC_WEPKEY:
			kid = (u_int) ireq->i_val;
			if (kid >= IEEE80211_WEP_NKID)
				return (EINVAL);

			len = sc->sc_key[kid].wk_len;

			if (priv_check(curthread, PRIV_NET80211_GETKEY) == 0)
				bcopy(sc->sc_key[kid].wk_key, key, len);
			else
				bzero(key, len);

			ireq->i_len = len;
			err = copyout(key, ireq->i_data, len);
		break;
		case IEEE80211_IOC_WEPTXKEY:
			ireq->i_val = sc->sc_def_txkey;
		break;
		case IEEE80211_IOC_WPA:
			switch (sc->sc_iflags & IEEE80211_F_WPA) {
  			case IEEE80211_F_WPA1:
				ireq->i_val = 1;
			break;
			case IEEE80211_F_WPA2:
				ireq->i_val = 2;
			break;
			case IEEE80211_F_WPA1 | IEEE80211_F_WPA2:
				ireq->i_val = 3;
			break;
			default:
				ireq->i_val = 0;
			}
		break;
		case IEEE80211_IOC_WPAKEY:
		break;
		case IEEE80211_IOC_CHANLIST:
			if (ireq->i_len > sizeof(sc->sc_chanlist))
				ireq->i_len = sizeof(sc->sc_chanlist);

			err = copyout(&sc->sc_chanlist, ireq->i_data,
			    ireq->i_len);
		break;
		case IEEE80211_IOC_SCANVALID:
			ireq->i_val = sc->sc_scanvalid / hz;
		break;
		case IEEE80211_IOC_SCAN_RESULTS:
			mtx_lock(&sc->sc_mtx);
			err = jsta_ioctl_scan_results(sc, ireq);
			mtx_unlock(&sc->sc_mtx);
		break;
		}
	break;
	case SIOCS80211:
		switch (ireq->i_type) {
		case IEEE80211_IOC_SSID:
			if ((ireq->i_val != 0) ||
			    (ireq->i_len > IEEE80211_NWID_LEN))
				return (EINVAL);

			err = copyin(ireq->i_data, tmpstr, ireq->i_len);
			if (err)
				break;

			memcpy(sc->sc_ssid, tmpstr, ireq->i_len);
			sc->sc_ssid_len = ireq->i_len;
		break;
		case IEEE80211_IOC_STATIONNAME:
			if ((ireq->i_val != 0) ||
			    (ireq->i_len > IEEE80211_NWID_LEN))
				return (EINVAL);

			err = copyin(ireq->i_data, tmpstr, ireq->i_len);
			if (err)
				break;

			memcpy(sc->sc_stationname, tmpstr, ireq->i_len);
			sc->sc_stationname_len = ireq->i_len;
		break;
		case IEEE80211_IOC_BSSID:
			if (ireq->i_len != sizeof(bssid))
				return (EINVAL);

			err = copyin(ireq->i_data, bssid, ireq->i_len);
			if (err)
				break;

			IEEE80211_ADDR_COPY(sc->sc_des_bssid, bssid);

			if (IEEE80211_ADDR_EQ(sc->sc_des_bssid, zerobssid))
				sc->sc_iflags &= ~IEEE80211_F_DESBSSID;
			else
				sc->sc_iflags |= IEEE80211_F_DESBSSID;
		break;
		case IEEE80211_IOC_AUTHMODE:
		break;
		case IEEE80211_IOC_PRIVACY:
			if (ireq->i_val)
 				sc->sc_iflags |= IEEE80211_F_PRIVACY;
			else
				sc->sc_iflags &= ~IEEE80211_F_PRIVACY;
		break;
		case IEEE80211_IOC_WEP:
		break;
		case IEEE80211_IOC_WEPKEY:
		break;
		case IEEE80211_IOC_WEPTXKEY:
		break;
		case IEEE80211_IOC_WPA:
		break;
		case IEEE80211_IOC_WPAKEY:
		break;
		case IEEE80211_IOC_DELKEY:
		break;
		case IEEE80211_IOC_CHANLIST:
		break;
		case IEEE80211_IOC_SCANVALID:
			if (ireq->i_val < IEEE80211_SCAN_VALID_MIN)
				err = EINVAL;
			else
				sc->sc_scanvalid = ireq->i_val * hz;
		break;
		case IEEE80211_IOC_SCAN_REQ:
			if ((ifp->if_drv_flags & IFF_DRV_RUNNING) == 0)
				return (ENXIO);

			if (ireq->i_len != sizeof(struct ieee80211_scan_req))
				return (EINVAL);

			err = copyin(ireq->i_data, &sr, sizeof(sr));
			if (err)
				break;

			mtx_lock(&sc->sc_mtx);
			err = jsta_scan(sc, &sr);
			mtx_unlock(&sc->sc_mtx);
		break;
		case IEEE80211_IOC_SCAN_CANCEL:
		break;
		case IEEE80211_IOC_MLME:
		break;
		}
	break;
	default:
		err = ether_ioctl(ifp, cmd, data);
	}

	return (err);
}

static int
jsta_media_change(struct ifnet *ifp)
{
	return (0);
}

static void
jsta_media_status(struct ifnet *ifp, struct ifmediareq *imr)
{
	imr->ifm_active = IFM_IEEE80211;
}

static void
jsta_event_scan_completed(struct jsta_softc *sc)
{
	struct ifnet *ifp = sc->sc_ifp;
	int err;

	mtx_lock(&sc->sc_mtx);

	err = jsta_get_scan_results(sc);
	if (err) {
		device_printf(sc->sc_dev,
		    "could not get scan results err=%d\n", err);
		goto done;
	}

	CURVNET_SET(ifp->if_vnet);
	rt_ieee80211msg(ifp, RTM_IEEE80211_SCAN, NULL, 0);
	CURVNET_RESTORE();

done:

	mtx_unlock(&sc->sc_mtx);
}

static void
jsta_event_connected(struct jsta_softc *sc)
{
	struct ifnet *ifp = sc->sc_ifp;

	mtx_lock(&sc->sc_mtx);

	sc->sc_associated = 1;

	CURVNET_SET(ifp->if_vnet);
	rt_ieee80211msg(ifp, RTM_IEEE80211_ASSOC, NULL, 0);
	if_link_state_change(ifp, LINK_STATE_UP);
	CURVNET_RESTORE();

	mtx_unlock(&sc->sc_mtx);
}

static void
jsta_event_disconnected(struct jsta_softc *sc)
{
	struct ifnet *ifp = sc->sc_ifp;

	mtx_lock(&sc->sc_mtx);

	sc->sc_associated = 0;

	CURVNET_SET(ifp->if_vnet);
	rt_ieee80211msg(ifp, RTM_IEEE80211_DISASSOC, NULL, 0);
	if_link_state_change(ifp, LINK_STATE_DOWN);
	CURVNET_RESTORE();

	mtx_unlock(&sc->sc_mtx);
}

static int
jsta_event_handler(struct jpt_event_listener *self,
    struct ps3_eurus_event *event)
{
	struct jsta_softc *sc = self->el_arg;

	switch (event->hdr.type) {
	case PS3_EURUS_EVENT_TYPE_0x40:
		switch (event->hdr.id) {
		case PS3_EURUS_EVENT_DEAUTH:
			jsta_event_disconnected(sc);
		break;
		}
	break;
	case PS3_EURUS_EVENT_TYPE_0x80:
		switch (event->hdr.id) {
		case PS3_EURUS_EVENT_SCAN_COMPLETED:
			jsta_event_scan_completed(sc);
		break;
		case PS3_EURUS_EVENT_CONNECTED:
		case PS3_EURUS_EVENT_WPA_CONNECTED:
			jsta_event_connected(sc);
		break;
		case PS3_EURUS_EVENT_BEACON_LOST:
		break;
		}
	break;
	}

	return (0);
}

static int
jsta_scan(struct jsta_softc *sc, struct ieee80211_scan_req *sr)
{
	struct ps3_eurus_cmd_start_scan *eurus_cmd_start_scan;
	unsigned char *buf, *ie;
	unsigned int payload_len, status;
	int i, chan;
	int err = 0;

	mtx_assert(&sc->sc_mtx, MA_OWNED);

	if (sr->sr_nssid > IEEE80211_SCAN_MAX_SSID)
		sr->sr_nssid = IEEE80211_SCAN_MAX_SSID;

	for (i = 0; i < sr->sr_nssid; i++) {
		if (sr->sr_ssid[i].len > IEEE80211_NWID_LEN)
			return (EINVAL);
	}

	buf = malloc(JSTA_CMD_BUFSZ, M_TEMP, M_WAIT | M_ZERO);
	if (!buf)
		return (ENOMEM);

	eurus_cmd_start_scan = (struct ps3_eurus_cmd_start_scan *) buf;
	eurus_cmd_start_scan->unknown2 =
	    (sr->sr_flags & IEEE80211_IOC_SCAN_ACTIVE) != 0;
	le16enc(&eurus_cmd_start_scan->channel_dwell, 0x64);

	ie = eurus_cmd_start_scan->ie;
	ie[0] = IEEE80211_ELEMID_DSPARMS;
	ie[1] = 0;

	for (i = 1, chan = 0; i < IEEE80211_CHAN_MAX; i++) {
		if (isset(sc->sc_chanlist, i)) {
			ie[2 + chan] = i;
			chan++;
		}
	}

	ie[1] = chan;
	payload_len = ie + 2 + ie[1] - (uint8_t *) eurus_cmd_start_scan;

	if (sr->sr_nssid > 0) {
		ie += 2 + ie[1];
		ie[0] = IEEE80211_ELEMID_SSID;
		ie[1] = sr->sr_ssid[0].len;
		memcpy(ie + 2, sr->sr_ssid[0].ssid, sr->sr_ssid[0].len);
	
		payload_len += 2 + ie[1];
	}

	err = jpt_exec_eurus_cmd(PS3_EURUS_CMD_START_SCAN,
	    eurus_cmd_start_scan, payload_len, &status, NULL, NULL);
	if (err)
		goto done;

	if (status != PS3_EURUS_CMD_OK) {
		err = EIO;
		goto done;
	}

done:

	free(buf, M_TEMP);

	return (err);
}

static void
jsta_free_scan_results(struct jsta_softc *sc)
{
	struct jsta_scan_result *sr, *sr_tmp;

	mtx_assert(&sc->sc_mtx, MA_OWNED);

	TAILQ_FOREACH_SAFE(sr, &sc->sc_srq, sr_queue, sr_tmp) {
		TAILQ_REMOVE(&sc->sc_srq, sr, sr_queue);
		free(sr, M_JSTA);
	}
}

static int
jsta_get_scan_results(struct jsta_softc *sc)
{
	struct ps3_eurus_cmd_get_scan_results *eurus_cmd_get_scan_results;
	struct ps3_eurus_scan_result *eurus_scan_result;
	struct jsta_scan_result *sr;
	unsigned char *buf, *ie;
	unsigned int status, resp_len;
	unsigned int eurus_scan_result_len, ie_len;
	int i;
	int err = 0;

	mtx_assert(&sc->sc_mtx, MA_OWNED);

	buf = malloc(JSTA_CMD_BUFSZ, M_TEMP, M_WAIT | M_ZERO);
	if (!buf)
		return (ENOMEM);

	eurus_cmd_get_scan_results =
	    (struct ps3_eurus_cmd_get_scan_results *) buf;

	err = jpt_exec_eurus_cmd(PS3_EURUS_CMD_GET_SCAN_RESULTS,
	    eurus_cmd_get_scan_results, PS3_EURUS_SCAN_RESULTS_MAXSIZE,
	    &status, &resp_len, eurus_cmd_get_scan_results);
	if (err)
		goto done;

	if (status != PS3_EURUS_CMD_OK) {
		err = EIO;
		goto done;
	}

	jsta_free_scan_results(sc);

	for (i = 0, eurus_scan_result = eurus_cmd_get_scan_results->result;
	     i < eurus_cmd_get_scan_results->count; i++) {
		eurus_scan_result_len = le16dec(&eurus_scan_result->length) +
		    sizeof(eurus_scan_result->length);
		ie_len = (uint8_t *) eurus_scan_result +
		    eurus_scan_result_len - eurus_scan_result->ie;

		sr = malloc(sizeof(*sr) + ie_len, M_JSTA, M_WAIT | M_ZERO);
		if (!sr)
			goto next;

		sr->sr_rssi = eurus_scan_result->rssi;
		sr->sr_capinfo = le16dec(&eurus_scan_result->capability);
		sr->sr_intval = le16dec(&eurus_scan_result->beacon_period);

		IEEE80211_ADDR_COPY(sr->sr_bssid, eurus_scan_result->bssid);

		sr->sr_ie_len = ie_len;
		memcpy(sr->sr_ie, eurus_scan_result->ie, ie_len);

		for (ie = sr->sr_ie; ie < (sr->sr_ie + sr->sr_ie_len);
		     ie += (2 + ie[1])) {
			switch (ie[0]) {
			case IEEE80211_ELEMID_SSID:
				sr->sr_ie_ssid = ie;
			break;
			case IEEE80211_ELEMID_DSPARMS:
				sr->sr_ie_dsparms = ie;
			break;
			case IEEE80211_ELEMID_RATES:
				sr->sr_ie_rates = ie;
			break;
			case IEEE80211_ELEMID_XRATES:
				sr->sr_ie_xrates = ie;
			break;
			case IEEE80211_ELEMID_ERP:
				sr->sr_ie_erp = ie;
			break;
			}
		}

		KASSERT(sr->sr_ie_ssid != NULL, ("ssid ie not found"));
		KASSERT(sr->sr_ie_dsparms != NULL,
		    ("ds parameter set ie not found"));
		KASSERT((sr->sr_ie_dsparms[2] >= 1) &&
			(sr->sr_ie_dsparms[2] <= 14),
		    ("invalid ds parameter set ie"));
		KASSERT(sr->sr_ie_rates != NULL, ("rates ie not found"));

		TAILQ_INSERT_TAIL(&sc->sc_srq, sr, sr_queue);

	next:
		/* move to next scan result */

		eurus_scan_result = (struct ps3_eurus_scan_result *)
		    ((uint8_t *) eurus_scan_result + eurus_scan_result_len);
	}

done:

	free(buf, M_TEMP);

	return (err);
}

static int
jsta_ioctl_chaninfo(struct jsta_softc *sc, struct ieee80211req *ireq)
{
	struct ieee80211req_chaninfo *ci;
	int len, i, chan;
	int err = 0;

	len = IEEE80211_CHANINFO_SIZE(sc->sc_nchans * 2);

	ci = malloc(len, M_TEMP, M_WAIT | M_ZERO);
	if (!ci)
		return (ENOMEM);

	ci->ic_nchans = sc->sc_nchans * 2;

	for (i = 1, chan = 0; i <= 14; i++) {
		if ((sc->sc_chan_info & (1 << (i - 1))) == 0)
			continue;

		ci->ic_chans[chan].ic_freq = ieee80211_ieee2mhz(i,
		    IEEE80211_CHAN_2GHZ);
		ci->ic_chans[chan].ic_ieee = i;
		ci->ic_chans[chan].ic_flags = IEEE80211_CHAN_B;
		chan++;

		ci->ic_chans[chan].ic_freq = ieee80211_ieee2mhz(i,
		    IEEE80211_CHAN_2GHZ);
		ci->ic_chans[chan].ic_ieee = i;
		ci->ic_chans[chan].ic_flags = IEEE80211_CHAN_G;
		chan++;
	}

	if (len > ireq->i_len)
		len = ireq->i_len;

	err = copyout(ci, ireq->i_data, len);

	free(ci, M_TEMP);

	return (err);
}

static int
jsta_ioctl_scan_results(struct jsta_softc *sc, struct ieee80211req *ireq)
{
	struct jsta_scan_result *sr;
	struct ieee80211req_scan_result *isr;
	unsigned char *buf;
	int nr;
	int nxr = 0;
	int len = 0;
	int err = 0;

	mtx_assert(&sc->sc_mtx, MA_OWNED);

	TAILQ_FOREACH(sr, &sc->sc_srq, sr_queue) {
		len += roundup(sizeof(*isr) + sr->sr_ie_ssid[1] +
		    sr->sr_ie_len, sizeof(uint32_t));
	}

	buf = malloc(len, M_TEMP, M_WAIT | M_ZERO);
	if (!buf)
		return (ENOMEM);

	isr = (struct ieee80211req_scan_result *) buf;

	TAILQ_FOREACH(sr, &sc->sc_srq, sr_queue) {
		isr->isr_ssid_len = sr->sr_ie_ssid[1];
		isr->isr_ie_len = sr->sr_ie_len;
		isr->isr_len = roundup(sizeof(*isr) +
		    isr->isr_ssid_len + isr->isr_ie_len, sizeof(uint32_t));
		isr->isr_ie_off = sizeof(struct ieee80211req_scan_result);
		isr->isr_freq = ieee80211_ieee2mhz(sr->sr_ie_dsparms[2],
		    IEEE80211_CHAN_2GHZ);
		isr->isr_rssi = sr->sr_rssi;
		isr->isr_capinfo = sr->sr_capinfo;
		isr->isr_intval = sr->sr_intval;

		if (sr->sr_ie_erp)
			isr->isr_erp = sr->sr_ie_erp[2];

		IEEE80211_ADDR_COPY(isr->isr_bssid, sr->sr_bssid);
		memcpy((uint8_t *) isr + isr->isr_ie_off,
		    sr->sr_ie_ssid + 2, isr->isr_ssid_len);
		memcpy((uint8_t *) isr + isr->isr_ie_off + isr->isr_ssid_len,
		    sr->sr_ie, isr->isr_ie_len);

		nr = min(sr->sr_ie_rates[1], IEEE80211_RATE_MAXSIZE);
		memcpy(isr->isr_rates, sr->sr_ie_rates + 2, nr);

		if (sr->sr_ie_xrates) {
			nxr = min(sr->sr_ie_xrates[1], IEEE80211_RATE_MAXSIZE - nr);
			memcpy(isr->isr_rates + nr, sr->sr_ie_xrates + 2, nxr);
		}

		isr->isr_nrates = nr + nxr;

		isr = (struct ieee80211req_scan_result *) ((uint8_t *) isr +
		    isr->isr_len);
	}

	if (len > ireq->i_len)
		len = ireq->i_len;

	ireq->i_len = len;

	err = copyout(buf, ireq->i_data, ireq->i_len);

	free(buf, M_TEMP);

	return (err);
}

static int
jsta_get_mac_addr(struct jsta_softc *sc)
{
	struct ps3_eurus_cmd_get_mac_addr_list *eurus_cmd_get_mac_addr_list;
	unsigned char *buf;
	unsigned int status, resp_len;
	int err = 0;

	buf = malloc(JSTA_CMD_BUFSZ, M_TEMP, M_WAIT | M_ZERO);
	if (!buf)
		return (ENOMEM);

	/* get MAC address list */

	eurus_cmd_get_mac_addr_list =
	    (struct ps3_eurus_cmd_get_mac_addr_list *) buf;

	err = jpt_exec_eurus_cmd(PS3_EURUS_CMD_GET_MAC_ADDR_LIST,
	    eurus_cmd_get_mac_addr_list, PS3_EURUS_MAC_ADDR_LIST_MAXSIZE,
	    &status, &resp_len, eurus_cmd_get_mac_addr_list);
	if (err)
		goto done;

	if (status != PS3_EURUS_CMD_OK) {
		err = EIO;
		goto done;
	}

	/* use first MAC address */

	memcpy(sc->sc_mac_addr, eurus_cmd_get_mac_addr_list->mac_addr,
	    ETHER_ADDR_LEN);

done:

	free(buf, M_TEMP);

	return (err);
}

static int
jsta_set_mac_addr(struct jsta_softc *sc,
    const uint8_t mac_addr[ETHER_ADDR_LEN])
{
	struct ps3_eurus_cmd_set_mac_addr *eurus_cmd_set_mac_addr;
	struct ps3_eurus_cmd_0x115b *eurus_cmd_0x115b;
	unsigned char *buf;
	unsigned int status;
	int err = 0;

	buf = malloc(JSTA_CMD_BUFSZ, M_TEMP, M_WAIT);
	if (!buf)
		return (ENOMEM);

	eurus_cmd_set_mac_addr = (struct ps3_eurus_cmd_set_mac_addr *) buf;
	memset(eurus_cmd_set_mac_addr, 0,
	    sizeof(*eurus_cmd_set_mac_addr));
	memcpy(eurus_cmd_set_mac_addr->mac_addr,
	    mac_addr, ETHER_ADDR_LEN);

	err = jpt_exec_eurus_cmd(PS3_EURUS_CMD_SET_MAC_ADDR,
	    eurus_cmd_set_mac_addr, sizeof(*eurus_cmd_set_mac_addr),
	    &status, NULL, NULL);
	if (err)
		goto done;

	if (status != PS3_EURUS_CMD_OK) {
		err = EIO;
		goto done;
	}

	eurus_cmd_0x115b = (struct ps3_eurus_cmd_0x115b *) buf;
	memset(eurus_cmd_0x115b, 0, sizeof(*eurus_cmd_0x115b));
	le16enc(&eurus_cmd_0x115b->unknown1, 0x1);
	le16enc(&eurus_cmd_0x115b->unknown2, 0x0);
	memcpy(eurus_cmd_0x115b->mac_addr, mac_addr, ETHER_ADDR_LEN);

	err = jpt_exec_eurus_cmd(PS3_EURUS_CMD_0x115b,
	    eurus_cmd_0x115b, sizeof(*eurus_cmd_0x115b),
	    &status, NULL, NULL);
	if (err)
		goto done;

	if (status != PS3_EURUS_CMD_OK) {
		err = EIO;
		goto done;
	}

done:

	free(buf, M_TEMP);

	return (err);
}

static int
jsta_get_chan_info(struct jsta_softc *sc)
{
	struct ps3_eurus_cmd_get_channel_info *eurus_cmd_get_channel_info;
	unsigned char *buf;
	unsigned int status, resp_len;
	int i;
	int err = 0;

	buf = malloc(JSTA_CMD_BUFSZ, M_TEMP, M_WAIT | M_ZERO);
	if (!buf)
		return (ENOMEM);

	eurus_cmd_get_channel_info =
	    (struct ps3_eurus_cmd_get_channel_info *) buf;

	err = jpt_exec_eurus_cmd(PS3_EURUS_CMD_GET_CHANNEL_INFO,
	    eurus_cmd_get_channel_info, sizeof(*eurus_cmd_get_channel_info),
	    &status, &resp_len, eurus_cmd_get_channel_info);
	if (err)
		goto done;

	if (status != PS3_EURUS_CMD_OK) {
		err = EIO;
		goto done;
	}

	sc->sc_chan_info = eurus_cmd_get_channel_info->channel_info;

	for (i = 1, sc->sc_nchans = 0; i <= 14; i++) {
		if (sc->sc_chan_info & (1 << (i - 1)))
			sc->sc_nchans++;
	}

done:

	free(buf, M_TEMP);

	return (err);
}

static device_method_t jsta_methods[] = {
	DEVMETHOD(device_probe, jsta_match),
	DEVMETHOD(device_attach, jsta_attach),
	DEVMETHOD(device_detach, jsta_detach),
	{ 0, 0 }
};

static driver_t jsta_driver = {
	.name = "jsta",
	.methods = jsta_methods,
	.size = sizeof(struct jsta_softc)
};

static devclass_t jsta_devclass;

DRIVER_MODULE(jsta, uhub, jsta_driver, jsta_devclass, NULL, 0);
MODULE_DEPEND(jsta, usb, 1, 1, 1);
MODULE_DEPEND(jsta, wlan, 1, 1, 1);
MODULE_DEPEND(jsta, jpt, 1, 1, 1);
MODULE_VERSION(jsta, 1);
