/*-
 * Copyright (c) 2013 glevand <geoffrey.levand@mail.ru>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $FreeBSD$
 */

#include <sys/cdefs.h>
__FBSDID("$FreeBSD$");

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/module.h>
#include <sys/lock.h>
#include <sys/mutex.h>
#include <sys/bio.h>
#include <sys/sbuf.h>
#include <sys/sysctl.h>
#include <sys/malloc.h>
#include <sys/kthread.h>
#include <sys/proc.h>
#include <sys/sched.h>
#include <sys/endian.h>
#include <geom/geom.h>

#include "g_bswap16.h"

SYSCTL_DECL(_kern_geom);
static SYSCTL_NODE(_kern_geom, OID_AUTO, bswap16, CTLFLAG_RW, 0, "GEOM_BSWAP16 stuff");
static u_int g_bswap16_debug = 0;
SYSCTL_UINT(_kern_geom_bswap16, OID_AUTO, debug, CTLFLAG_RW, &g_bswap16_debug, 0,
    "Debug level");

static int g_bswap16_destroy(struct g_geom *gp, boolean_t force);
static int g_bswap16_destroy_geom(struct gctl_req *req, struct g_class *mp,
    struct g_geom *gp);
static void g_bswap16_config(struct gctl_req *req, struct g_class *mp,
    const char *verb);

struct g_class g_bswap16_class = {
	.name = G_BSWAP16_CLASS_NAME,
	.version = G_VERSION,
	.ctlreq = g_bswap16_config,
	.destroy_geom = g_bswap16_destroy_geom
};

static void
g_bswap16_orphan(struct g_consumer *cp)
{
	g_topology_assert();

	g_bswap16_destroy(cp->geom, 1);
}

static void
g_bswap16_read_done(struct bio *bp)
{
	struct g_bswap16_softc *sc;
	struct g_provider *pp;
	struct bio *pbp;

	G_BSWAP16_LOGREQ(2, bp, "Request done.");

	pbp = bp->bio_parent;
	pp = pbp->bio_to;
	sc = pp->geom->softc;

	mtx_lock(&sc->sc_bioq_lock);
	bioq_insert_tail(&sc->sc_bioq, bp);
	mtx_unlock(&sc->sc_bioq_lock);
	wakeup(sc);
}

static void
g_bswap16_start(struct bio *bp)
{
	struct g_bswap16_softc *sc;
	struct g_geom *gp;
	struct g_provider *pp;
	struct g_consumer *cp;
	struct bio *cbp;

	gp = bp->bio_to->geom;
	sc = gp->softc;
	pp = LIST_FIRST(&gp->provider);
	cp = LIST_FIRST(&gp->consumer);

	KASSERT(pp != NULL, ("NULL pp"));

	G_BSWAP16_LOGREQ(2, bp, "Request received.");

	cbp = g_clone_bio(bp);
	if (!cbp) {
		g_io_deliver(bp, ENOMEM);
		return;
	}

	cbp->bio_done = g_std_done;
	cbp->bio_to = pp;

	if (bp->bio_cmd == BIO_WRITE) {
		mtx_lock(&sc->sc_bioq_lock);
		bioq_insert_tail(&sc->sc_bioq, cbp);
		mtx_unlock(&sc->sc_bioq_lock);
		wakeup(sc);
	} else {
		if (bp->bio_cmd == BIO_READ)
			cbp->bio_done = g_bswap16_read_done;

		G_BSWAP16_LOGREQ(2, cbp, "Sending request.");

		g_io_request(cbp, cp);
	}
}

static int
g_bswap16_access(struct g_provider *pp, int dr, int dw, int de)
{
	struct g_geom *gp;
	struct g_consumer *cp;
	int error;

	gp = pp->geom;
	cp = LIST_FIRST(&gp->consumer);

	error = g_access(cp, dr, dw, de);

	return (error);
}

static void
g_bswap16_worker(void *arg)
{
	struct g_bswap16_softc *sc;
	struct g_geom *gp;
	struct g_consumer *cp;
	struct g_provider *pp;
	struct bio *bp, *pbp;
	int i;

	sc = arg;

	for (;;) {
		mtx_lock(&sc->sc_bioq_lock);
		bp = bioq_takefirst(&sc->sc_bioq);
		if (!bp) {
			if (sc->sc_flags & G_BSWAP16_FLAG_DESTROY) {
				G_BSWAP16_DEBUG(1, "Thread %s exiting.",
				    curthread->td_proc->p_comm);
				sc->sc_flags |= G_BSWAP16_FLAG_WORKER_DONE;
				wakeup(sc);
				mtx_unlock(&sc->sc_bioq_lock);
				kproc_exit(0);
			}

			msleep(sc, &sc->sc_bioq_lock, PDROP,
			    "g_bswap16:wait", 0);
			continue;
		}
		mtx_unlock(&sc->sc_bioq_lock);

		pbp = bp->bio_parent;
		pp = pbp->bio_to;
		gp = pp->geom;
		cp = LIST_FIRST(&gp->consumer);

		/* swap bytes */

		for (i = 0; i < pbp->bio_length / 2; i++)
			*((uint16_t *) pbp->bio_data + i) =
			    bswap16(*((uint16_t *) pbp->bio_data + i));

		if (pbp->bio_cmd == BIO_READ) {
			if (!pbp->bio_error)
				pbp->bio_error = bp->bio_error;

			g_destroy_bio(bp);

			if (pbp->bio_error) {
				pbp->bio_completed = 0;
				g_io_deliver(pbp, pbp->bio_error);
				continue;
			}

			pbp->bio_completed = pbp->bio_length;
			g_io_deliver(pbp, pbp->bio_error);
		} else {
			G_BSWAP16_LOGREQ(2, bp, "Sending request.");

			g_io_request(bp, cp);
		}
	}
}

static int
g_bswap16_create(struct gctl_req *req, struct g_class *mp, struct g_provider *pp)
{
	struct g_bswap16_softc *sc;
	struct g_geom *gp;
	struct g_provider *newpp;
	struct g_consumer *cp;
	char name[64];
	int error;

	g_topology_assert();

	gp = NULL;
	newpp = NULL;
	cp = NULL;

	snprintf(name, sizeof(name), "%s%s", pp->name, G_BSWAP16_SUFFIX);

	LIST_FOREACH(gp, &mp->geom, geom) {
		if (!strcmp(gp->name, name)) {
			gctl_error(req, "Provider %s already exists.", name);
			return (EEXIST);
		}
	}

	sc = g_malloc(sizeof(*sc), M_WAITOK | M_ZERO);

	bioq_init(&sc->sc_bioq);
	mtx_init(&sc->sc_bioq_lock, "g_bswap16:bioq", NULL, MTX_DEF);

	error = kproc_create(g_bswap16_worker, sc, &sc->sc_proc, 0, 0,
	    "g_bswap16 %s", pp->name);
	if (error) {
		gctl_error(req, "Cannot create kernel thread for %s (error=%d).",
		    pp->name, error);
		goto fail;
	}

	gp = g_new_geomf(mp, "%s", name);
	gp->softc = sc;
	gp->start = g_bswap16_start;
	gp->orphan = g_bswap16_orphan;
	gp->access = g_bswap16_access;

	newpp = g_new_providerf(gp, "%s", gp->name);
	newpp->mediasize = pp->mediasize;
	newpp->sectorsize = pp->sectorsize;

	cp = g_new_consumer(gp);
	error = g_attach(cp, pp);
	if (error) {
		gctl_error(req, "Cannot attach to provider %s.", pp->name);
		goto fail;
	}

	g_error_provider(newpp, 0);

	G_BSWAP16_DEBUG(0, "Device %s created.", gp->name);

	return (error);

fail:

	mtx_lock(&sc->sc_bioq_lock);
	sc->sc_flags |= G_BSWAP16_FLAG_DESTROY;
	wakeup(sc);
	while (!(sc->sc_flags & G_BSWAP16_FLAG_WORKER_DONE)) {
		msleep(sc, &sc->sc_bioq_lock, PRIBIO,
		    "g_bswap16:destroy", 0);
	}
	mtx_destroy(&sc->sc_bioq_lock);

	if (cp->provider)
		g_detach(cp);

	g_destroy_consumer(cp);
	g_destroy_provider(newpp);
	g_free(gp->softc);
	g_destroy_geom(gp);

	return (error);
}

static int
g_bswap16_destroy(struct g_geom *gp, boolean_t force)
{
	struct g_bswap16_softc *sc;
	struct g_provider *pp;

	g_topology_assert();

	if (!gp->softc)
		return (ENXIO);

	sc = gp->softc;

	pp = LIST_FIRST(&gp->provider);
	if (pp && (pp->acr || pp->acw || pp->ace)) {
		if (force) {
			G_BSWAP16_DEBUG(0, "Device %s is still open, so it "
			    "can't be definitely removed.", pp->name);
		} else {
			G_BSWAP16_DEBUG(1, "Device %s is still open (r%dw%de%d).",
			    pp->name, pp->acr, pp->acw, pp->ace);
			return (EBUSY);
		}
	} else {
		G_BSWAP16_DEBUG(0, "Device %s removed.", gp->name);
	}

	mtx_lock(&sc->sc_bioq_lock);
	sc->sc_flags |= G_BSWAP16_FLAG_DESTROY;
	wakeup(sc);
	while (!(sc->sc_flags & G_BSWAP16_FLAG_WORKER_DONE)) {
		msleep(sc, &sc->sc_bioq_lock, PRIBIO,
		    "g_bswap16:destroy", 0);
	}
	mtx_destroy(&sc->sc_bioq_lock);

	g_free(gp->softc);
	gp->softc = NULL;

	g_wither_geom(gp, ENXIO);

	return (0);
}

static int
g_bswap16_destroy_geom(struct gctl_req *req, struct g_class *mp, struct g_geom *gp)
{
	return (g_bswap16_destroy(gp, 0));
}

static void
g_bswap16_ctl_create(struct gctl_req *req, struct g_class *mp)
{
	struct g_provider *pp;
	int i, *nargs;
	const char *name;
	char param[16];
	int error;

	g_topology_assert();

	nargs = gctl_get_paraml(req, "nargs", sizeof(*nargs));
	if (!nargs) {
		gctl_error(req, "No '%s' argument", "nargs");
		return;
	}

	if (*nargs <= 0) {
		gctl_error(req, "Missing device(s).");
		return;
	}

	for (i = 0; i < *nargs; i++) {
		snprintf(param, sizeof(param), "arg%d", i);

		name = gctl_get_asciiparam(req, param);
		if (!name) {
			gctl_error(req, "No 'arg%d' argument", i);
			return;
		}

		if (!strncmp(name, "/dev/", strlen("/dev/")))
			name += strlen("/dev/");

		pp = g_provider_by_name(name);
		if (!pp) {
			gctl_error(req, "Provider %s is invalid.", name);
			return;
		}

		error = g_bswap16_create(req, mp, pp);
		if (error)
			return;
	}
}

static struct g_geom *
g_bswap16_find_geom(struct g_class *mp, const char *name)
{
	struct g_geom *gp;

	LIST_FOREACH(gp, &mp->geom, geom) {
		if (!strcmp(gp->name, name))
			return (gp);
	}

	return (NULL);
}

static void
g_bswap16_ctl_destroy(struct gctl_req *req, struct g_class *mp)
{
	struct g_geom *gp;
	int *nargs, *force;
	const char *name;
	char param[16];
	int i;
	int error;

	g_topology_assert();

	nargs = gctl_get_paraml(req, "nargs", sizeof(*nargs));
	if (!nargs) {
		gctl_error(req, "No '%s' argument", "nargs");
		return;
	}

	if (*nargs <= 0) {
		gctl_error(req, "Missing device(s).");
		return;
	}

	force = gctl_get_paraml(req, "force", sizeof(*force));
	if (!force) {
		gctl_error(req, "No 'force' argument");
		return;
	}

	for (i = 0; i < *nargs; i++) {
		snprintf(param, sizeof(param), "arg%d", i);

		name = gctl_get_asciiparam(req, param);
		if (!name) {
			gctl_error(req, "No 'arg%d' argument", i);
			return;
		}

		if (!strncmp(name, "/dev/", strlen("/dev/")))
			name += strlen("/dev/");

		gp = g_bswap16_find_geom(mp, name);
		if (!gp) {
			gctl_error(req, "Device %s is invalid.", name);
			return;
		}

		error = g_bswap16_destroy(gp, *force);
		if (error) {
			gctl_error(req, "Cannot destroy device %s (error=%d).",
			    gp->name, error);
			return;
		}
	}
}

static void
g_bswap16_config(struct gctl_req *req, struct g_class *mp, const char *verb)
{
	uint32_t *version;

	g_topology_assert();

	version = gctl_get_paraml(req, "version", sizeof(*version));
	if (!version) {
		gctl_error(req, "No '%s' argument.", "version");
		return;
	}
	if (*version != G_BSWAP16_VERSION) {
		gctl_error(req, "Userland and kernel parts are out of sync.");
		return;
	}

	if (!strcmp(verb, "create")) {
		g_bswap16_ctl_create(req, mp);
		return;
	} else if (!strcmp(verb, "destroy")) {
		g_bswap16_ctl_destroy(req, mp);
		return;
	}

	gctl_error(req, "Unknown verb.");
}

DECLARE_GEOM_CLASS(g_bswap16_class, g_bswap16);
