/*-
 * Copyright (c) 2013 glevand <geoffrey.levand@mail.ru>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $FreeBSD$
 */

#include <sys/cdefs.h>
__FBSDID("$FreeBSD$");

#include <sys/param.h>
#include <sys/systm.h>
#include <sys/kernel.h>
#include <sys/module.h>
#include <sys/uio.h>
#include <sys/lock.h>
#include <sys/mutex.h>
#include <sys/bio.h>
#include <sys/sbuf.h>
#include <sys/sysctl.h>
#include <sys/malloc.h>
#include <sys/kthread.h>
#include <sys/proc.h>
#include <sys/sched.h>
#include <sys/endian.h>
#include <geom/geom.h>
#include <opencrypto/cryptodev.h>

#include "g_aes_xts.h"

MALLOC_DEFINE(M_AES_XTS, "aes-xts data", "GEOM_AES_XTS data");

SYSCTL_DECL(_kern_geom);
static SYSCTL_NODE(_kern_geom, OID_AUTO, aes_xts, CTLFLAG_RW, 0, "GEOM_AES_XTS stuff");
static u_int g_aes_xts_debug = 0;
SYSCTL_UINT(_kern_geom_aes_xts, OID_AUTO, debug, CTLFLAG_RW, &g_aes_xts_debug, 0,
    "Debug level");

static int g_aes_xts_destroy(struct g_geom *gp, boolean_t force);
static int g_aes_xts_destroy_geom(struct gctl_req *req, struct g_class *mp,
    struct g_geom *gp);
static void g_aes_xts_config(struct gctl_req *req, struct g_class *mp,
    const char *verb);

struct g_class g_aes_xts_class = {
	.name = G_AES_XTS_CLASS_NAME,
	.version = G_VERSION,
	.ctlreq = g_aes_xts_config,
	.destroy_geom = g_aes_xts_destroy_geom
};

static void
g_aes_xts_orphan(struct g_consumer *cp)
{
	g_topology_assert();

	g_aes_xts_destroy(cp->geom, TRUE);
}

static void
g_aes_xts_read_done(struct bio *bp)
{
	struct g_aes_xts_softc *sc;
	struct g_provider *pp;
	struct bio *pbp;

	G_AES_XTS_LOGREQ(2, bp, "Request done.");

	pbp = bp->bio_parent;
	pp = pbp->bio_to;
	sc = pp->geom->softc;

	mtx_lock(&sc->sc_bioq_lock);
	bioq_insert_tail(&sc->sc_bioq, bp);
	mtx_unlock(&sc->sc_bioq_lock);
	wakeup(sc);
}

static void
g_aes_xts_write_done(struct bio *bp)
{
	struct g_aes_xts_softc *sc;
	struct g_provider *pp;
	struct bio *pbp;

	G_AES_XTS_LOGREQ(2, bp, "Request done.");

	pbp = bp->bio_parent;
	pp = pbp->bio_to;
	sc = pp->geom->softc;

	free(bp->bio_driver2, M_AES_XTS);
	bp->bio_driver2 = NULL;

	if (bp->bio_error) {
		G_AES_XTS_LOGREQ(0, bp, "Crypto WRITE request failed (error=%d).",
		    bp->bio_error);

		bp->bio_completed = 0;
	}

	if (!pbp->bio_error)
		pbp->bio_error = bp->bio_error;

	pbp->bio_completed = bp->bio_completed;

	g_destroy_bio(bp);
	g_io_deliver(pbp, pbp->bio_error);
}

static void
g_aes_xts_start(struct bio *bp)
{
	struct g_aes_xts_softc *sc;
	struct g_geom *gp;
	struct g_provider *pp;
	struct g_consumer *cp;
	struct bio *cbp;

	gp = bp->bio_to->geom;
	sc = gp->softc;
	pp = LIST_FIRST(&gp->provider);
	cp = LIST_FIRST(&gp->consumer);

	KASSERT(pp != NULL, ("NULL pp"));

	G_AES_XTS_LOGREQ(2, bp, "Request received.");

	switch (bp->bio_cmd) {
	case BIO_READ:
	case BIO_WRITE:
	case BIO_GETATTR:
	case BIO_FLUSH:
	break;
	case BIO_DELETE:
	default:
		g_io_deliver(bp, EOPNOTSUPP);
		return;
	}

	cbp = g_clone_bio(bp);
	if (!cbp) {
		g_io_deliver(bp, ENOMEM);
		return;
	}

	cbp->bio_done = g_std_done;
	cbp->bio_to = pp;

	if (bp->bio_cmd == BIO_WRITE) {
		mtx_lock(&sc->sc_bioq_lock);
		bioq_insert_tail(&sc->sc_bioq, cbp);
		mtx_unlock(&sc->sc_bioq_lock);
		wakeup(sc);
	} else {
		if (bp->bio_cmd == BIO_READ)
			cbp->bio_done = g_aes_xts_read_done;

		G_AES_XTS_LOGREQ(2, cbp, "Sending request.");

		g_io_request(cbp, cp);
	}
}

static int
g_aes_xts_access(struct g_provider *pp, int dr, int dw, int de)
{
	struct g_geom *gp;
	struct g_consumer *cp;
	int error;

	gp = pp->geom;
	cp = LIST_FIRST(&gp->consumer);

	error = g_access(cp, dr, dw, de);

	return (error);
}

static void
g_aes_xts_dumpconf(struct sbuf *sb, const char *indent, struct g_geom *gp,
    struct g_consumer *cp, struct g_provider *pp)
{
	struct g_aes_xts_softc *sc;
	int i;

	g_topology_assert();

	sc = gp->softc;
	if (sc == NULL)
		return;

	if (pp || cp)
		return;

	sbuf_printf(sb, "%s<KeyLength>%u</KeyLength>\n", indent,
	    sc->sc_keylen);

	sbuf_printf(sb, "%s<Key>", indent);
	for (i = 0; i < (sc->sc_keylen >> 3); i++)
		sbuf_printf(sb, "%02x", sc->sc_key[i]);
	sbuf_printf(sb, "</Key>\n");
}

static int
g_aes_xts_crypto_read_done(struct cryptop *crp)
{
	struct bio *bp, *pbp;
	int error;

	if (crp->crp_etype == EAGAIN) {
		G_AES_XTS_DEBUG(1, "Rerunning crypto READ request.");

		crp->crp_etype = 0;

		error = crypto_dispatch(crp);
		if (!error)
			return (0);

		crp->crp_etype = error;
	}

	bp = (struct bio *) crp->crp_opaque;
	pbp = bp->bio_parent;

	bp->bio_inbed++;

	if (!crp->crp_etype) {
		G_AES_XTS_DEBUG(3, "Crypto READ request done (%d/%d).",
		    bp->bio_inbed, bp->bio_children);

		bp->bio_completed += crp->crp_olen;
	} else {
		G_AES_XTS_DEBUG(1, "Crypto READ request failed (%d/%d) error=%d.",
		    bp->bio_inbed, bp->bio_children, crp->crp_etype);

		if (!bp->bio_error)
			bp->bio_error = crp->crp_etype;
	}

	if (bp->bio_inbed < bp->bio_children)
		return (0);

	/* We have all sectors now */

	free(bp->bio_driver2, M_AES_XTS);
	bp->bio_driver2 = NULL;

	if (bp->bio_error) {
		G_AES_XTS_LOGREQ(0, bp, "Crypto READ request failed (error=%d).",
		    bp->bio_error);

		bp->bio_completed = 0;
	}

	if (!pbp->bio_error)
		pbp->bio_error = bp->bio_error;

	pbp->bio_completed = bp->bio_completed;

	g_destroy_bio(bp);
	g_io_deliver(pbp, pbp->bio_error);

	return (0);
}

static int
g_aes_xts_crypto_write_done(struct cryptop *crp)
{
	struct bio *bp, *pbp;
	struct g_geom *gp;
	struct g_consumer *cp;
	int error;

	if (crp->crp_etype == EAGAIN) {
		G_AES_XTS_DEBUG(1, "Rerunning crypto WRITE request.");

		crp->crp_etype = 0;

		error = crypto_dispatch(crp);
		if (!error)
			return (0);

		crp->crp_etype = error;
	}

	bp = (struct bio *) crp->crp_opaque;
	pbp = bp->bio_parent;
	gp = pbp->bio_to->geom;
	cp = LIST_FIRST(&gp->consumer);

	bp->bio_inbed++;

	if (!crp->crp_etype) {
		G_AES_XTS_DEBUG(3, "Crypto WRITE request done (%d/%d).",
		    bp->bio_inbed, bp->bio_children);

		bp->bio_completed += crp->crp_olen;
	} else {
		G_AES_XTS_DEBUG(1, "Crypto WRITE request failed (%d/%d) error=%d.",
		    bp->bio_inbed, bp->bio_children, crp->crp_etype);

		if (!bp->bio_error)
			bp->bio_error = crp->crp_etype;
	}

	if (bp->bio_inbed < bp->bio_children)
		return (0);

	/* We have all sectors now */

	if (bp->bio_error) {
		G_AES_XTS_LOGREQ(0, bp, "Crypto WRITE request failed (error=%d).",
		    bp->bio_error);

		free(bp->bio_driver2, M_AES_XTS);
		bp->bio_driver2 = NULL;

		bp->bio_completed = 0;

		if (!pbp->bio_error)
			pbp->bio_error = bp->bio_error;

		pbp->bio_completed = bp->bio_completed;

		g_destroy_bio(bp);
		g_io_deliver(pbp, pbp->bio_error);

		return (0);
	}

	G_AES_XTS_LOGREQ(2, bp, "Sending request.");

	bp->bio_data = bp->bio_driver2;
	bp->bio_done = g_aes_xts_write_done;

	g_io_request(bp, cp);

	return (0);
}

static void
g_aes_xts_crypto_run(struct bio *bp)
{
	struct g_aes_xts_softc *sc;
	struct bio *pbp;
	struct g_geom *gp;
	struct g_provider *pp;
	struct cryptop *crp;
	struct cryptodesc *crd;
	struct uio *uio;
	struct iovec *iov;
	u_int i, nsec, secsize;
	size_t size;
	off_t off;
	u_char *p, *data;
	uint8_t offset[8];
	int err, error;

	pbp = bp->bio_parent;
	gp = pbp->bio_to->geom;
	sc = gp->softc;
	pp = LIST_FIRST(&gp->provider);

	secsize = pp->sectorsize;
	nsec = bp->bio_length / secsize;

	size = sizeof(*crp) * nsec;
	size += sizeof(*crd) * nsec;
	size += sizeof(*uio) * nsec;
	size += sizeof(*iov) * nsec;

	if (bp->bio_cmd == BIO_WRITE)
		size += bp->bio_length;

	p = malloc(size, M_AES_XTS, M_WAITOK);

	bp->bio_inbed = 0;
	bp->bio_children = nsec;
	bp->bio_driver2 = p;
	bp->bio_completed = 0;

	if (bp->bio_cmd == BIO_READ) {
		data = bp->bio_data;
	} else {
		data = p;
		p += bp->bio_length;
		bcopy(bp->bio_data, data, bp->bio_length);
	}

	/* Every sector is encrypted/decrypted separately */

	error = 0;

	for (i = 0, off = bp->bio_offset; i < nsec; i++, off += secsize) {
		crp = (struct cryptop *) p;
		p += sizeof(*crp);

		crd = (struct cryptodesc *) p;
		p += sizeof(*crd);

		uio = (struct uio *) p;
		p += sizeof(*uio);

		iov = (struct iovec *) p;
		p += sizeof(*iov);

		iov->iov_len = secsize;
		iov->iov_base = data;
		data += secsize;

		uio->uio_iov = iov;
		uio->uio_iovcnt = 1;
		uio->uio_segflg = UIO_SYSSPACE;
		uio->uio_resid = secsize;

		crp->crp_sid = sc->sc_crypto_sid;
		crp->crp_ilen = secsize;
		crp->crp_olen = secsize;
		crp->crp_opaque = (void *) bp;
		crp->crp_buf = (void *) uio;

		if (bp->bio_cmd == BIO_READ)
			crp->crp_callback = g_aes_xts_crypto_read_done;
		else
			crp->crp_callback = g_aes_xts_crypto_write_done;

		crp->crp_flags = CRYPTO_F_IOV | CRYPTO_F_CBIFSYNC | CRYPTO_F_REL;
		crp->crp_desc = crd;

		crd->crd_skip = 0;
		crd->crd_len = secsize;
		crd->crd_flags = CRD_F_IV_EXPLICIT | CRD_F_IV_PRESENT | CRD_F_KEY_EXPLICIT;

		if (bp->bio_cmd == BIO_WRITE)
			crd->crd_flags |= CRD_F_ENCRYPT;

		crd->crd_alg = CRYPTO_AES_XTS;
		crd->crd_key = sc->sc_key;
		crd->crd_klen = sc->sc_keylen;

		/* Generate IV */

		le64enc(offset, (uint64_t) off);
		bcopy(offset, crd->crd_iv, sizeof(offset));
		bzero(crd->crd_iv + sizeof(offset), sizeof(crd->crd_iv) - sizeof(offset));

		crd->crd_next = NULL;
		crp->crp_etype = 0;

		err = crypto_dispatch(crp);
		if (!error)
			error = err;
	}

	if (!bp->bio_error)
		bp->bio_error = error;
}

static void
g_aes_xts_worker(void *arg)
{
	struct g_aes_xts_softc *sc;
	struct bio *bp, *pbp;

	sc = arg;

	for (;;) {
		mtx_lock(&sc->sc_bioq_lock);
		bp = bioq_takefirst(&sc->sc_bioq);
		if (!bp) {
			if (sc->sc_flags & G_AES_XTS_FLAG_DESTROY) {
				G_AES_XTS_DEBUG(1, "Thread %s exiting.",
				    curthread->td_proc->p_comm);
				sc->sc_flags |= G_AES_XTS_FLAG_WORKER_DONE;
				wakeup(sc);
				mtx_unlock(&sc->sc_bioq_lock);
				kproc_exit(0);
			}

			msleep(sc, &sc->sc_bioq_lock, PDROP,
			    "g_aes_xts:wait", 0);
			continue;
		}
		mtx_unlock(&sc->sc_bioq_lock);

		pbp = bp->bio_parent;

		if (pbp->bio_cmd == BIO_READ) {
			if (!pbp->bio_error)
				pbp->bio_error = bp->bio_error;

			if (pbp->bio_error) {
				g_destroy_bio(bp);
				pbp->bio_completed = 0;
				g_io_deliver(pbp, pbp->bio_error);
				continue;
			}
		}

		g_aes_xts_crypto_run(bp);
	}
}

static int
g_aes_xts_create(struct gctl_req *req, struct g_class *mp, struct g_provider *pp,
    const u_char *key, int keysize)
{
	struct g_aes_xts_softc *sc;
	struct g_geom *gp;
	struct g_provider *newpp;
	struct g_consumer *cp;
	struct cryptoini crie;
	char name[64];
	int error;

	g_topology_assert();

	gp = NULL;
	newpp = NULL;
	cp = NULL;

	snprintf(name, sizeof(name), "%s%s", pp->name, G_AES_XTS_SUFFIX);

	LIST_FOREACH(gp, &mp->geom, geom) {
		if (!strcmp(gp->name, name)) {
			gctl_error(req, "Provider %s already exists.", name);
			return (EEXIST);
		}
	}

	sc = g_malloc(sizeof(*sc), M_WAITOK | M_ZERO);

	bcopy(key, sc->sc_key, keysize);
	sc->sc_keylen = keysize << 3;

	bzero(&crie, sizeof(crie));
	crie.cri_alg = CRYPTO_AES_XTS;
	crie.cri_klen = sc->sc_keylen;
	crie.cri_key = sc->sc_key;

	error = crypto_newsession(&sc->sc_crypto_sid, &crie, CRYPTOCAP_F_SOFTWARE);
	if (error) {
		gctl_error(req, "Cannot set up crypto session "
		    "for %s (error=%d).", pp->name, error);
		goto fail;
	}

	bioq_init(&sc->sc_bioq);
	mtx_init(&sc->sc_bioq_lock, "g_aes_xts:bioq", NULL, MTX_DEF);

	error = kproc_create(g_aes_xts_worker, sc, &sc->sc_proc, 0, 0,
	    "g_aes_xts %s", pp->name);
	if (error) {
		gctl_error(req, "Cannot create kernel thread for %s (error=%d).",
		    pp->name, error);
		goto fail;
	}

	gp = g_new_geomf(mp, "%s", name);
	gp->softc = sc;
	gp->start = g_aes_xts_start;
	gp->orphan = g_aes_xts_orphan;
	gp->access = g_aes_xts_access;
	gp->dumpconf = g_aes_xts_dumpconf;

	newpp = g_new_providerf(gp, "%s", gp->name);
	newpp->mediasize = pp->mediasize;
	newpp->sectorsize = pp->sectorsize;

	cp = g_new_consumer(gp);
	error = g_attach(cp, pp);
	if (error) {
		gctl_error(req, "Cannot attach to provider %s.", pp->name);
		goto fail;
	}

	g_error_provider(newpp, 0);

	G_AES_XTS_DEBUG(0, "Device %s created.", gp->name);

	return (error);

fail:

	mtx_lock(&sc->sc_bioq_lock);
	sc->sc_flags |= G_AES_XTS_FLAG_DESTROY;
	wakeup(sc);
	while (!(sc->sc_flags & G_AES_XTS_FLAG_WORKER_DONE)) {
		msleep(sc, &sc->sc_bioq_lock, PRIBIO,
		    "g_aes_xts:destroy", 0);
	}
	mtx_destroy(&sc->sc_bioq_lock);

	if (cp->provider)
		g_detach(cp);

	g_destroy_consumer(cp);
	g_destroy_provider(newpp);

	if (sc->sc_crypto_sid)
		crypto_freesession(sc->sc_crypto_sid);

	g_free(gp->softc);
	g_destroy_geom(gp);

	return (error);
}

static int
g_aes_xts_destroy(struct g_geom *gp, boolean_t force)
{
	struct g_aes_xts_softc *sc;
	struct g_provider *pp;

	g_topology_assert();

	if (!gp->softc)
		return (ENXIO);

	sc = gp->softc;

	pp = LIST_FIRST(&gp->provider);
	if (pp && (pp->acr || pp->acw || pp->ace)) {
		if (force) {
			G_AES_XTS_DEBUG(0, "Device %s is still open, so it "
			    "can't be definitely removed.", pp->name);
		} else {
			G_AES_XTS_DEBUG(1, "Device %s is still open (r%dw%de%d).",
			    pp->name, pp->acr, pp->acw, pp->ace);
			return (EBUSY);
		}
	} else {
		G_AES_XTS_DEBUG(0, "Device %s removed.", gp->name);
	}

	mtx_lock(&sc->sc_bioq_lock);
	sc->sc_flags |= G_AES_XTS_FLAG_DESTROY;
	wakeup(sc);
	while (!(sc->sc_flags & G_AES_XTS_FLAG_WORKER_DONE)) {
		msleep(sc, &sc->sc_bioq_lock, PRIBIO,
		    "g_aes_xts:destroy", 0);
	}
	mtx_destroy(&sc->sc_bioq_lock);

	crypto_freesession(sc->sc_crypto_sid);

	g_free(gp->softc);
	gp->softc = NULL;

	g_wither_geom(gp, ENXIO);

	return (0);
}

static int
g_aes_xts_destroy_geom(struct gctl_req *req, struct g_class *mp, struct g_geom *gp)
{
	return (g_aes_xts_destroy(gp, FALSE));
}

static void
g_aes_xts_ctl_create(struct gctl_req *req, struct g_class *mp)
{
	struct g_provider *pp;
	int *nargs, keysize;
	u_char *key;
	const char *name;
	int error;

	g_topology_assert();

	nargs = gctl_get_paraml(req, "nargs", sizeof(*nargs));
	if (!nargs) {
		gctl_error(req, "No '%s' argument", "nargs");
		return;
	}

	if (*nargs != 1) {
		gctl_error(req, "Invalid number of arguments.");
		return;
	}

	key = gctl_get_param(req, "key", &keysize);
	if ((key == NULL) || ((keysize != G_AES_XTS_128_KEYSIZE) &&
	    (keysize != G_AES_XTS_256_KEYSIZE))) {
		gctl_error(req, "No '%s' argument.", "key");
		return;
	}

	name = gctl_get_asciiparam(req, "arg0");
	if (!name) {
		gctl_error(req, "No '%s' argument.", "arg0");
		return;
	}

	if (!strncmp(name, "/dev/", strlen("/dev/")))
		name += strlen("/dev/");

	pp = g_provider_by_name(name);
	if (!pp) {
		gctl_error(req, "Provider %s is invalid.", name);
		return;
	}

	error = g_aes_xts_create(req, mp, pp, key, keysize);
	if (error)
		return;
}

static struct g_geom *
g_aes_xts_find_geom(struct g_class *mp, const char *name)
{
	struct g_geom *gp;

	LIST_FOREACH(gp, &mp->geom, geom) {
		if (!strcmp(gp->name, name))
			return (gp);
	}

	return (NULL);
}

static void
g_aes_xts_ctl_destroy(struct gctl_req *req, struct g_class *mp)
{
	struct g_geom *gp;
	int *nargs, *force;
	const char *name;
	char param[16];
	int i;
	int error;

	g_topology_assert();

	nargs = gctl_get_paraml(req, "nargs", sizeof(*nargs));
	if (!nargs) {
		gctl_error(req, "No '%s' argument", "nargs");
		return;
	}

	if (*nargs <= 0) {
		gctl_error(req, "Missing device(s).");
		return;
	}

	force = gctl_get_paraml(req, "force", sizeof(*force));
	if (!force) {
		gctl_error(req, "No 'force' argument");
		return;
	}

	for (i = 0; i < *nargs; i++) {
		snprintf(param, sizeof(param), "arg%d", i);

		name = gctl_get_asciiparam(req, param);
		if (!name) {
			gctl_error(req, "No 'arg%d' argument", i);
			return;
		}

		if (!strncmp(name, "/dev/", strlen("/dev/")))
			name += strlen("/dev/");

		gp = g_aes_xts_find_geom(mp, name);
		if (!gp) {
			gctl_error(req, "Device %s is invalid.", name);
			return;
		}

		error = g_aes_xts_destroy(gp, *force);
		if (error) {
			gctl_error(req, "Cannot destroy device %s (error=%d).",
			    gp->name, error);
			return;
		}
	}
}

static void
g_aes_xts_config(struct gctl_req *req, struct g_class *mp, const char *verb)
{
	uint32_t *version;

	g_topology_assert();

	version = gctl_get_paraml(req, "version", sizeof(*version));
	if (!version) {
		gctl_error(req, "No '%s' argument.", "version");
		return;
	}
	if (*version != G_AES_XTS_VERSION) {
		gctl_error(req, "Userland and kernel parts are out of sync.");
		return;
	}

	if (!strcmp(verb, "create")) {
		g_aes_xts_ctl_create(req, mp);
		return;
	} else if (!strcmp(verb, "destroy")) {
		g_aes_xts_ctl_destroy(req, mp);
		return;
	}

	gctl_error(req, "Unknown verb.");
}

DECLARE_GEOM_CLASS(g_aes_xts_class, g_aes_xts);
MODULE_DEPEND(g_aes_xts, crypto, 1, 1, 1);
