/*-
 * Copyright (c) 2013 glevand <geoffrey.levand@mail.ru>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHORS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $FreeBSD$
 */

#include <sys/cdefs.h>
__FBSDID("$FreeBSD$");

#include <stdio.h>
#include <stdint.h>
#include <string.h>
#include <assert.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <libgeom.h>
#include <core/geom.h>
#include <misc/subr.h>
#include <g_aes_xts.h>

uint32_t lib_version = G_LIB_VERSION;
uint32_t version = G_AES_XTS_VERSION;

static void aes_xts_create(struct gctl_req *req, unsigned int flags);

struct g_command class_commands[] = {
	{
		"create",
		G_FLAG_VERBOSE | G_FLAG_LOADKLD,
		aes_xts_create,
		{
			{ 'k', "keyfile", G_VAL_OPTIONAL, G_TYPE_STRING },
			G_OPT_SENTINEL
		},
		"[-v] -k keyfile prov"
	},
	{
		"destroy",
		G_FLAG_VERBOSE,
		NULL,
		{
			{ 'f', "force", NULL, G_TYPE_BOOL },
			G_OPT_SENTINEL
		},
		"[-fv] prov ..."
	},
	G_CMD_SENTINEL
};

static void
aes_xts_create(struct gctl_req *req, unsigned int flags)
{
	const char *prov;
	const char *file;
	unsigned char key[G_AES_XTS_256_KEYSIZE];
	ssize_t nread;
	int verbose = 0;
	int keylen;
	int nargs;
	int fd;
	int error;

	if ((flags & G_FLAG_VERBOSE) != 0)
		verbose = 1;

	nargs = gctl_get_int(req, "nargs");
	if (nargs != 1) {
		gctl_error(req, "Invalid number of arguments.");
		return;
	}

	prov = gctl_get_ascii(req, "arg0");

	file = gctl_get_ascii(req, "keyfile");
	assert(file != NULL);

	if (!strcmp(file, "-")) {
		fd = STDIN_FILENO;
	} else {
		fd = open(file, O_RDONLY);
		if (fd == -1) {
			gctl_error(req, "Cannot open %s: %s.",
			    file, strerror(errno));
			return;
		}
	}

	nread = 0;
	keylen = 0;

	do {
		nread = read(fd, key + keylen, G_AES_XTS_256_KEYSIZE - keylen);
		if (nread == -1) {
			gctl_error(req, "Cannot read %s: %s.",
			    file, strerror(errno));
			if (strcmp(file, "-"))
				close(fd);
			return;
		}

		keylen += nread;
	} while (nread && (keylen < G_AES_XTS_256_KEYSIZE));

	if (strcmp(file, "-"))
		close(fd);

	if ((keylen != G_AES_XTS_128_KEYSIZE) && (keylen != G_AES_XTS_256_KEYSIZE)) {
		gctl_error(req, "Invalid key length.");
		return;
	}

	gctl_ro_param(req, "key", keylen, key);

	if (gctl_issue(req) == NULL) {
		if (verbose)
			printf("Attached to %s.\n", prov);
	}
}
