#include "types.h"
#include "spu.h"
#include "netrpc.h"
#include "util.h"

#define _EIEIO(ctxt) netrpc_eieio(ctxt)

void spu_slb_invalidate_all(netrpc_ctxt_t *ctxt, u64 spu_priv2)
{
	s32 i;
	
	write_u64(ctxt, SPU_RADDR(spu_priv2, _OFFSET_SLB_Invalidate_All), 0);
	
	_EIEIO(ctxt);
	
	for(i = 0; i < SPU_SLB_MAX_ENTRIES; i++)
	{
		write_u64(ctxt, SPU_RADDR(spu_priv2, _OFFSET_SLB_Index), i);
		write_u64(ctxt, SPU_RADDR(spu_priv2, _OFFSET_SLB_VSID), 0);
		write_u64(ctxt, SPU_RADDR(spu_priv2, _OFFSET_SLB_ESID), 0);
		
		_EIEIO(ctxt);
	}
}

s32 spu_slb_set_entry(netrpc_ctxt_t *ctxt, u64 spu_priv2, u64 index, u64 esid, u64 vsid)
{
	if(index >= SPU_SLB_MAX_ENTRIES)
		return -1;
	
	write_u64(ctxt, SPU_RADDR(spu_priv2, _OFFSET_SLB_Index), index);
	write_u64(ctxt, SPU_RADDR(spu_priv2, _OFFSET_SLB_VSID), vsid);
	
	_EIEIO(ctxt);
	
	write_u64(ctxt, SPU_RADDR(spu_priv2, _OFFSET_SLB_ESID), esid);
	
	_EIEIO(ctxt);
	
	return 0;
}

void spu_in_mbox_write(netrpc_ctxt_t *ctxt, u64 spu_problem, u32 val)
{
	write_u32(ctxt, SPU_RADDR(spu_problem, _OFFSET_SPU_In_Mbox), val);
	
	_EIEIO(ctxt);
}

void spu_in_mbox_write_64(netrpc_ctxt_t *ctxt, u64 spu_problem, u64 val)
{
	write_u32(ctxt, SPU_RADDR(spu_problem, _OFFSET_SPU_In_Mbox), val >> 32);
	write_u32(ctxt, SPU_RADDR(spu_problem, _OFFSET_SPU_In_Mbox), val & 0xffffffff);
	
	_EIEIO(ctxt);
}

void spu_sig_notify_1_2_write_64(netrpc_ctxt_t *ctxt, u64 spu_problem, u64 val)
{
	write_u32(ctxt, SPU_RADDR(spu_problem, _OFFSET_SPU_Sig_Notify_1), val >> 32);
	write_u32(ctxt, SPU_RADDR(spu_problem, _OFFSET_SPU_Sig_Notify_2), val & 0xffffffff);
	
	_EIEIO(ctxt);
}

void spu_isolation_req_enable(netrpc_ctxt_t *ctxt, u64 spu_priv2)
{
	write_u64(ctxt, SPU_RADDR(spu_priv2, _OFFSET_SPU_PrivCntl), 0x4);
	
	_EIEIO(ctxt);
}

void spu_isolation_req(netrpc_ctxt_t *ctxt, u64 spu_problem)
{
	write_u32(ctxt, SPU_RADDR(spu_problem, _OFFSET_SPU_RunCntl), SPU_RunCntl_ISOLATION_REQ);
	
	_EIEIO(ctxt);
}

void spu_stop_req(netrpc_ctxt_t *ctxt, u64 spu_problem)
{
	write_u32(ctxt, SPU_RADDR(spu_problem, _OFFSET_SPU_RunCntl), SPU_RunCntl_STOP_REQ);
	
	_EIEIO(ctxt);
}

void spu_runcntl_req(netrpc_ctxt_t *ctxt, u64 spu_problem, u32 req)
{
	write_u32(ctxt, SPU_RADDR(spu_problem, _OFFSET_SPU_RunCntl), req);
	
	_EIEIO(ctxt);
}

u8 spu_mbox_stat_intr_out_mbox_count(netrpc_ctxt_t *ctxt, u64 spu_problem)
{
	u32 v;
	read_u32(ctxt, SPU_RADDR(spu_problem, _OFFSET_SPU_Mbox_Stat), &v);
	return (v >> 16) & 0xff;
}

u8 spu_mbox_stat_in_mbox_count(netrpc_ctxt_t *ctxt, u64 spu_problem)
{
	u32 v;
	read_u32(ctxt, SPU_RADDR(spu_problem, _OFFSET_SPU_Mbox_Stat), &v);
	return (v >> 8) & 0xff;
}

u8 spu_mbox_stat_out_mbox_count(netrpc_ctxt_t *ctxt, u64 spu_problem)
{
	u32 v;
	read_u32(ctxt, SPU_RADDR(spu_problem, _OFFSET_SPU_Mbox_Stat), &v);
	return v & 0xff;
}

u8 spu_mfc_cmd_exec(netrpc_ctxt_t *ctxt, u64 spu_problem, u32 lsa, u64 ea, u16 size, u16 tag, u16 classid, u16 cmd)
{
	write_u32(ctxt, SPU_RADDR(spu_problem, _OFFSET_MFC_LSA), lsa);
	write_u32(ctxt, SPU_RADDR(spu_problem, _OFFSET_MFC_EAH), ea >> 32);
	write_u32(ctxt, SPU_RADDR(spu_problem, _OFFSET_MFC_EAL), ea & 0xffffffff);
	write_u32(ctxt, SPU_RADDR(spu_problem, _OFFSET_MFC_Size), (size << 16) | tag);
	write_u32(ctxt, SPU_RADDR(spu_problem, _OFFSET_MFC_ClassID_CMD), (classid << 16) | cmd);
	
	_EIEIO(ctxt);
	
	u32 v;
	read_u32(ctxt, SPU_RADDR(spu_problem, _OFFSET_MFC_CMDStatus), &v);
	return v & 0x3;
}

u8 spu_mfc_cmd_tag_status(netrpc_ctxt_t *ctxt, u64 spu_problem, u8 tag)
{
	write_u32(ctxt, SPU_RADDR(spu_problem, _OFFSET_Prxy_QueryType), 0);
	write_u32(ctxt, SPU_RADDR(spu_problem, _OFFSET_Prxy_QueryMask), 1 << tag);
	
	_EIEIO(ctxt);
	
	u32 v;
	read_u32(ctxt, SPU_RADDR(spu_problem, _OFFSET_Prxy_TagStatus), &v);
	return (v >> tag) & 0x1;
}
