#ifndef _NETRPC_H_
#define _NETRPC_H_

#include "types.h"

/*! Arguments count. */
#define _ARGCNT(args) _ARRAY_CNT(args, u64)

/*! Common constants. */
#define PS3_U64_SIZE 8
#define NETRPC_PORT 1337
#define MAX_READMEM 1024
#define MAX_WRITEMEM 1024

/*! Netrpc commands. */
#define RPC_PING 0
#define RPC_READMEM 1
#define RPC_WRITEMEM 2
#define RPC_HVCALL 3
#define RPC_ADDMMIO 4
#define RPC_DELMMIO 5
#define RPC_CLRMMIO 6
#define RPC_MEMSET 7
#define RPC_VECTOR 8
#define RPC_SYNC 9
#define RPC_CALL 10
#define RPC_EIEIO 11

/*! Netrpc vector. */
struct rpc_vec {
	u64 vec0;
	u64 vec1;
	u64 copy_dst;
	u64 copy_src;
	u32 copy_size;
};

/*! Netrpc header. */
struct rpc_header {
	u32 cmd;
	u32 tag;
	union {
		struct {
			u64 addr;
			u32 size;
			u8 data[];
		} memop;
		struct {
			u32 numin, numout;
			u64 code;
			u64 regs[];
		} hvcall;
		struct {
			u64 start;
			u32 size;
		} addmmio;
		struct {
			u64 start;
		} delmmio;
		struct {
			s64 retcode;
			u8 retdata[];
		} reply;
		struct rpc_vec vector;
		struct {
			u64 addr;
			u32 size;
		} sync;
		struct {
			u64 addr;
			u64 args[8];
		} call;
	};
} __attribute__((packed));

/*! Netrpc context. */
typedef struct _netrpc_ctxt
{
	/*! Connection socket. */
	int socket;
	/*! Current packet tag. */
	u32 tag;
} netrpc_ctxt_t;

/*!
* \brief Connect to ps3 netrpc host.
*/
s32 netrpc_connect(netrpc_ctxt_t *ctxt, u32 ip, u32 port);

/*!
* \brief Close connection to ps3 netrpc host.
*/
void netrpc_close(netrpc_ctxt_t *ctxt);

/*!
* \brief Ping ps3 netrpc host.
*/
s64 netrpc_ping(netrpc_ctxt_t *ctxt);

/*!
* \brief Read memory from ps3 netrpc host.
*        (max. 1024 bytes)
*/
s64 netrpc_readmem(netrpc_ctxt_t *ctxt, u64 addr, void *buffer, u32 size, u32 *rdsize);

/*!
* \brief Memcpy from ps3 netrpc host.
*/
s64 netrpc_memcpy_in(netrpc_ctxt_t *ctxt, void *buffer, u64 addr, u32 length);

/*!
* \brief Write memory to ps3 netrpc host.
*        (max. 1024 bytes)
*/
s64 netrpc_writemem(netrpc_ctxt_t *ctxt, u64 addr, const void *buffer, u32 size);

/*!
* \brief Memcpy to ps3 netrpc host.
*/
s64 netrpc_memcpy_out(netrpc_ctxt_t *ctxt, u64 addr, void *buffer, u32 length);

/*!
* \brief Call hypervisor function on ps3 netrpc host.
*/
s64 netrpc_hvcall(netrpc_ctxt_t *ctxt, u64 code, const u64 arg_in[], u32 numin, u64 arg_out[], u32 numout);

/*!
* \brief Add mmio region on ps3 netrpc host.
*/
s64 netrpc_addmmio(netrpc_ctxt_t *ctxt, u64 start, u32 size);

/*!
* \brief Delete mmio region on ps3 netrpc host.
*/
s64 netrpc_delmmio(netrpc_ctxt_t *ctxt, u64 start);

/*!
* \brief Clear all mmio regions on ps3 netrpc host.
*/
s64 netrpc_clrmmio(netrpc_ctxt_t *ctxt);

/*!
* \brief TODO!
*/
s64 netrpc_memset(netrpc_ctxt_t *ctxt, u64 addr, u64 value);

/*!
* \brief TODO!
*/
s64 netrpc_vector(netrpc_ctxt_t *ctxt, u64 vec0, u64 vec1, u64 copy_dst, u64 copy_src, u32 copy_size);

/*!
* \brief Sync memory for execution on ps3 netrpc host.
*/
s64 netrpc_sync_before_exec(netrpc_ctxt_t *ctxt, u64 addr, u32 size);

/*!
* \brief Call function on ps3 netrpc host.
*/
s64 netrpc_call(netrpc_ctxt_t *ctxt, u64 addr, u64 arg1, u64 arg2, u64 arg3, u64 arg4, u64 arg5, u64 arg6, u64 arg7, u64 arg8);

/*!
* \brief Execute eieio instruction on ps3 netrpc host.
*/
s64 netrpc_eieio(netrpc_ctxt_t *ctxt);

#endif
