#line 1 "source/patching.c"
/*
 * patching.c
 *
 * Copyright (C) an0nym0us
 *
 * This software is distributed under the terms of the GNU General Public
 * License ("GPL") version 3, as published by the Free Software Foundation.
 *
 */

/*
 * includes
 */

#include <stdio.h>

#include <sys/thread.h>

#include "common.h"
#include "debug.h"
#include "patching.h"

#include "syncutil.h"
#include "patchutil.h"


/*
 * defines
 */

extern fwVersions fw_data ;
extern u32 log_fd ;
extern char string[MAX_STRING] ;


/*
 * functions
 */

/* patching initialization */
void
patchingInitialization ( s32 *patched, patchData *p_data, u64 *priority )
{
  dbgprintf ( "initializing" ) ;

  p_data->patched = patched ;

  /* initialize mutex attributes */
  p_data->mutex_attr.key = PATCH_KEY ;
  p_data->mutex_attr.attr_protocol = SYS_MUTEX_PROTOCOL_FIFO ;
  p_data->mutex_attr.attr_pshared = SYS_MUTEX_ATTR_PSHARED ;
  p_data->mutex_attr.attr_recursive = SYS_MUTEX_ATTR_RECURSIVE ;
  p_data->mutex_attr.attr_adaptive = SYS_MUTEX_ATTR_ADAPTIVE ;

  /* initialize condition attributes */
  p_data->cond_attr.key = PATCH_KEY ;
  p_data->cond_attr.attr_pshared = SYS_COND_ATTR_PSHARED ;

  dbgprintf( "create sync stuff" );

  /* create mutex */
  syncutilMutexCreate ( &p_data->mutex, &p_data->mutex_attr ) ;

  /* create cond */
  syncutilCondCreate ( &p_data->cond, &p_data->mutex, &p_data->cond_attr ) ;

  /* start menu thread */
  dbgprintf( "thread spawn" );
  patchingThread ( p_data, priority ) ;

  dbgprintf ( "initialized" ) ;
}

void
patchingFinish ( patchData *p_data )
{
  /* signal main thread */
  syncutilMutexLock ( &p_data->mutex, NO_TIMEOUT ) ;
  dbgprintf( "signalling done" );
  syncutilCondSignal ( &p_data->cond ) ;
  syncutilMutexUnlock ( &p_data->mutex ) ;

  /* destroy cond */
  syncutilCondDestroy ( &p_data->cond ) ;

  /* destroy mutex */
  syncutilMutexDestroy ( &p_data->mutex ) ;

  //free ( pdata->path ) ;
}

/* patching thread */
void
patchingThread ( patchData *p_data, u64 *priority )
{
  dbgprintf( "starting" ) ;

  sys_ppu_thread_t ppu_thread;

  /* lock patching mutex */
  syncutilMutexLock ( &p_data->mutex, NO_TIMEOUT ) ;

  /* create patching thread */
  dbgprintf ( "thread create" ) ;
  sysThreadCreate ( &ppu_thread, patchingWorker, p_data,
                    *priority += 10, ( size_t ) 0x100000, 0,
                    "patchingWorker" ) ;

  /* wait for patching thread */
  dbgprintf ( "thread waiting" ) ;
  syncutilCondWait ( &p_data->cond, NO_TIMEOUT ) ;

  /* release lock */
  syncutilMutexUnlock ( &p_data->mutex ) ;

  dbgprintf( "completed" ) ;
}

void
patchingWorker ( void *arg )
{
  dbgprintf( "thread started, patching" ) ;
  static s32 ret = 0 ;

  /* cast void *arg to patchData */
  patchData *patch_data = (patchData*) arg ;

  *patch_data->patched = 0 ;

  /* patchutil init */
  if ( ( ret = patchutilInitialization ( patch_data ) ) != 0 )
  {
    errprintf ( "fatal error: 0x%x", ret ) ;
    goto error ;
  }
  else
  {
    argprintf ( "patchutil initialize returned: 0x%x (%d)", ret, ret ) ;
  }

  /* verify lv2 patches */
  if ( patch_data->lv2_patch == 0 )
  {
    errprintf ( "ERROR: LV2 not patched: %d", patch_data->lv2_patch ) ;
    goto error ;
  }

  /* verify lv1 patches */
  if ( patch_data->lv1_patch == 0 )
  {
    errprintf ( "ERROR: LV1 not patched: %d", patch_data->lv1_patch ) ;
    goto error ;
  }

  /* patch LV1 */
  if ( ( ret = patchShit () ) != 0 )
  {
    errprintf ( "fatal error: 0x%x", ret ) ;
    goto error ;
  }
  else
  {
    argprintf ( "patchShit returned: 0x%x (%d)", ret, ret ) ;
  }

  *patch_data->patched = 1 ;

  dbgprintf( "patched successfully" ) ;

  goto done ;

error:
  dbgprintf( "patching failed!" ) ;

  *patch_data->patched = -1 ;

  syscall_lv2_beep ( SYSCALL_BEEP_THRICE ) ;

done:
  patchingFinish ( patch_data ) ;

  dbgprintf( "exiting" ) ;

  /* exit thread */
  sysThreadExit ( 0 ) ;
}

