#line 0 "source/dumping.c"
/*
 * dumping.c
 *
 * Copyright (C) an0nym0us
 *
 * This software is distributed under the terms of the GNU General Public
 * License ("GPL") version 3, as published by the Free Software Foundation.
 *
 */

/*
 * includes
 */

#include <stdio.h>
#include <unistd.h>

#include <sys/thread.h>

#include "common.h"
#include "debug.h"
#include "syncutil.h"
#include "dumping.h"
#include "dump_functions.h"


/*
 * globals
 */

extern fwVersions fw_data ;
extern u32 log_fd ;
extern char string[MAX_STRING] ;


/*
 * defines
 */


/*
 * functions
 */

/* dumping initialization */
void
dumpingInitialization ( dumpData *d_data, eventData *e_data, videoutilData *vu_data, u64 *priority )
{
  dbgprintf ( "initializing" ) ;

  d_data->row = 0 ;
  *d_data->action = 0 ;
  *d_data->dumped = 0 ;

  d_data->e_data = e_data ;
  d_data->exitapp = &e_data->exitapp ;

  d_data->vu_data = vu_data ;

  d_data->buffer_bitmap_ptr = NULL ;
  d_data->page_bitmap_ptr[0] = NULL ;
  d_data->page_bitmap_ptr[1] = NULL ;

  /* initialize semaphore attributes */
  d_data->sem_attr.key = DUMP_KEY ;
  d_data->sem_attr.attr_protocol = SYS_SEM_ATTR_PROTOCOL ;
  d_data->sem_attr.attr_pshared = SYS_SEM_ATTR_PSHARED ;

  /* initialize mutex attributes */
  d_data->mutex_attr.key = DUMP_KEY ;
  d_data->mutex_attr.attr_protocol = SYS_MUTEX_PROTOCOL_FIFO ;
  d_data->mutex_attr.attr_pshared = SYS_MUTEX_ATTR_PSHARED ;
  d_data->mutex_attr.attr_recursive = SYS_MUTEX_ATTR_RECURSIVE ;
  d_data->mutex_attr.attr_adaptive = SYS_MUTEX_ATTR_ADAPTIVE ;

  /* initialize reader/writer attributes */
  d_data->rwlock_attr.key = DUMP_KEY ;
  d_data->rwlock_attr.attr_protocol = SYS_RWLOCK_PROTOCOL_FIFO ;
  d_data->rwlock_attr.attr_pshared = SYS_RWLOCK_ATTR_PSHARED ;
  d_data->rwlock_attr.flags = 0 ;

  /* initialize condition attributes */
  d_data->cond_attr.key = DUMP_KEY ;
  d_data->cond_attr.attr_pshared = SYS_COND_ATTR_PSHARED ;

  /* menu events */
  d_data->mnu_event = &e_data->mnu_event ;
  d_data->mnu_queue = &e_data->mnu_queue ;
  d_data->mnu_queue_attr = &e_data->mnu_queue_attr ;
  d_data->mnu_port = &e_data->mnu_port ;

  /* dump events */
  d_data->dmp_event = &e_data->dmp_event ;
  d_data->dmp_queue = &e_data->dmp_queue ;
  d_data->dmp_queue_attr = &e_data->dmp_queue_attr ;
  d_data->dmp_port = &e_data->dmp_port ;

  /* image events */
  d_data->img_event = &e_data->img_event ;
  d_data->img_queue = &e_data->img_queue ;
  d_data->img_queue_attr = &e_data->img_queue_attr ;
  d_data->img_port = &e_data->img_port ;

  /* create semaphore */
  syncutilSemCreate ( &d_data->sem, &d_data->sem_attr, 1, SEM_CONSUMERS ) ;

  /* create reader/writer lock */
  syncutilRwLockCreate ( &d_data->rwlock, &d_data->rwlock_attr ) ;

  /* create mutex */
  syncutilMutexCreate ( &d_data->mutex, &d_data->mutex_attr ) ;

  /* create cond */
  syncutilCondCreate ( &d_data->cond, &d_data->mutex, &d_data->cond_attr ) ;

  /* start dump thread */
  dumpingThread ( d_data, priority ) ;

  dbgprintf ( "initialized" ) ;
}

void
dumpingFinish ( dumpData *d_data )
{
  dbgprintf ( "finishing" ) ;

  /* destroy semaphore */
  syncutilSemDestroy ( &d_data->sem ) ;

  /* destroy rw lock */
  syncutilRwLockDestroy ( &d_data->rwlock ) ;

  /* destroy lwcond */
  syncutilCondDestroy ( &d_data->cond ) ;

  /* destroy lwmutext */
  syncutilMutexDestroy ( &d_data->mutex ) ;

  //free ( pdata->path ) ;

  dbgprintf ( "finished" ) ;
}

/* dumping thread */
void
dumpingThread ( dumpData *d_data, u64 *priority )
{
  dbgprintf( "starting" ) ;

  /* lock dumping mutex */
  syncutilMutexLock ( &d_data->mutex, TIMEOUT ) ;

  /* create dumping thread */
  dbgprintf( "thread create" );
  sysThreadCreate ( &d_data->ppu_thread, dumpingWorker, ( void * ) d_data,
                    *priority += 10, ( size_t ) 0x100000, 0,
                    "dumpingWorker" ) ;

  /* wait for dumping thread to be ready */
  dbgprintf( "thread waiting" );
  syncutilCondWait ( &d_data->cond, NO_TIMEOUT ) ;

  /* release lock */
  syncutilMutexUnlock ( &d_data->mutex ) ;

  dbgprintf( "started" ) ;
}

void
dumpingWorker ( void *arg )
{
  s32 timeout = 0 ;

  dbgprintf( "starting" ) ;

  /* cast the void *arg to dumpData */
  dumpData *d_data = ( dumpData* ) arg ;

  aioData aio_data ;
  d_data->aio_data = &aio_data ;

  dbgprintf ( "signalling" ) ;

  /* signal main thread is ready */
  syncutilCondSignal ( &d_data->cond ) ;

  /* lock mutex */
  syncutilMutexLock ( &d_data->mutex, TIMEOUT ) ;

  dbgprintf ( "waiting" ) ;

  /* wait for main to be ready */
  syncutilCondWait ( &d_data->cond, NO_TIMEOUT ) ;

  /* release lock */
  syncutilMutexUnlock ( &d_data->mutex ) ;

  dbgprintf ( "entering loop" ) ;

  /* wait for patching */
  while ( *d_data->exitapp && *d_data->patched == 0 )
  {
    argprintf ( "waiting for patching: %d", timeout ) ;

    /* wait */
    usleep ( 500000 ) ;

    /* timeout after 5 loops */
    if ( timeout > 5 )
    {
      dbgprintf ( "timeout waiting for patching, did you read the readme.txt?" ) ;
      dumpingWriteExitapp ( d_data->e_data ) ;
      break ;
    }
    else
    {
      dbgprintf ( "patching done." ) ;
      timeout++;
    }
  }

  /* render frames until exit */
  while ( *d_data->exitapp && *d_data->patched == 1 )
  {
    /* dumps */
    dumpingCheckUpdate ( d_data ) ;
  }

  dbgprintf ( "left loop" ) ;

  dumpingFinish ( d_data ) ;

  dbgprintf ( "exiting" ) ;

  /* exit thread */
  sysThreadExit ( 0 ) ;
}

/* check for dump event */
void
dumpingCheckUpdate ( dumpData *d_data )
{
  dumpingCheckDumpEvent ( d_data ) ;
}

s32
dumpingCheckDumpEvent ( dumpData *d_data )
{
  s32 ret = -1 ;

  s32 dump = dumpingReceiveDumpEvent ( d_data ) ;

  switch ( dump )
  {
    case DUMP_EXIT:
      dumpingWriteExitapp ( d_data->e_data ) ;
      break ;
    case DUMP_NONE:
      ret = 0 ;
      break ;
    case DUMP_LV1:
      ret = dumpingLv1 ( d_data ) ;
      break ;
    case DUMP_LV2:
      ret = dumpingLv2 ( d_data ) ;
      break ;
    case DUMP_FLASH:
      ret = dumpingFlash ( d_data ) ;
      break ;
    case DUMP_eEID:
      ret = dumpingeEID ( d_data ) ;
      break ;
    case DUMP_MMIO_NOR_FLASH:
      break ;
    default:
      break ;
  }

  sleep ( 2 ) ;

  if ( *d_data->exitapp )
  {
    dumpingSendMainUpdate ( d_data ) ;
  }

  return ( ret ) ;
}

/* receive dump selection via event queue */
s32
dumpingReceiveDumpEvent ( dumpData *d_data )
{
  s32 ret = 0 ;

  eventutilReceiveEventQueue ( d_data->dmp_queue, d_data->dmp_event, NO_TIMEOUT ) ;

  if ( d_data->dmp_event->data_1 > 0 )
  {
    ret = d_data->dmp_event->data_1 ;
  }
  else if ( d_data->dmp_event->data_1 == DUMP_EXIT && d_data->dmp_event->data_2 == DUMP_EXIT && d_data->dmp_event->data_3 == DUMP_EXIT )
  {
    ret = DUMP_EXIT ;
  }

  argprintf ( "read queue: %x, source: %016"PRIx64", update: %016"PRIx64":%016"PRIx64":%016"PRIx64, *d_data->dmp_queue, d_data->dmp_event->source, d_data->dmp_event->data_1, d_data->dmp_event->data_2, d_data->dmp_event->data_3 ) ;

  return ( ret ) ;
}

inline void
dumpingWriteExitapp ( eventData *e_data )
{
  dbgprintf ( "exit toggle" ) ;

  syncutilRwLockWriteLock ( &e_data->rwlock, NO_TIMEOUT ) ;
  e_data->exitapp = 0 ;
  syncutilRwLockWriteUnlock ( &e_data->rwlock ) ;
}

