#line 1 "source/dump_flash.c"
/*
 * dump_flash.c
 *
 * Copyright (C) an0nym0us
 *
 * This software is distributed under the terms of the GNU General Public
 * License ("GPL") version 3, as published by the Free Software Foundation.
 *
 */

/*
 * includes
 */

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <inttypes.h>
#include <time.h>

#include <ppu_intrinsics.h>

#include <sys/time.h>

#include "common.h"
#include "debug.h"
#include "patchutil.h"
#include "hexdump.h"

#include "file.h"
#include "vflash.h"
#include "dumping.h"
#include "patching.h"


/*
 * defines
 */

#ifndef MAX_BUFFERS
#define MAX_BUFFERS       4
#endif


/*
 * globals
 */

extern fwVersions fw_data ;
extern u32 log_fd ;
extern char string[MAX_STRING] ;


/*
 * functions
 */

/* dump NAND flash */
s32
dump_flash ( s32 store, const char *filename, dumpData *d_data )
{
#define NSECTORS ( PAGE_SIZE / DUMP_FLASH_SECTOR_SIZE )

  aioData aio_data ;
  struct storage_device_info device_info ;
  u8 *usr_buffer_ptr = NULL ;
  s32 ret = -1 ;
  s32 mask = -1 ;
  u64 size = 0ULL ;
  u8 current_buffer = 0 ;

  u32 count = 0 ;
  s32 page_counter = 0 ;
  s32 file_counter = 0 ;
  s32 percent_pages = 0 ;
  s32 percent_counter = 0 ;
  s32 progress_counter = 1 ;
  s32 current_page = 0 ;

  time_t starttime = 0 ;
  time_t donetime = 0 ;
  double sectime = 0 ;
  u64 loopstart = 0 ;
  u64 loopdone = 0 ;
  u64 looptotal = 0 ;
  u64 loopavg = 0 ;
  u64 copystart = 0 ;
  u64 copydone = 0 ;
  u64 copytotal = 0 ;
  u64 copyavg = 0 ;
  u64 writestart = 0 ;
  u64 writedone = 0 ;
  u64 writetotal = 0 ;
  u64 writeavg = 0 ;

  s32 vflash = 0 ;
  s32 start_sector = DUMP_FLASH_START_SECTOR ;
  s32 sector_count = 0 ;
  s32 sector_size = DUMP_FLASH_SECTOR_SIZE ;
  s32 device_flags = DUMP_FLASH_FLAGS ;
  u32 device_handle = 0 ;
  u32 sectors_read = 0 ;
  u64 device_id = 0ULL ;

  imageBitmap *page_bitmap_ptr[MAX_BUFFERS] = { NULL, NULL, NULL, NULL } ;

  page_bitmap_ptr[0] = (imageBitmap*) d_data->page_bitmap_ptr[0] ;
  argprintf ( "page_bitmap_ptr[0]: %p, page_bitmap_ptr[0]->ptr: %p", page_bitmap_ptr[0], page_bitmap_ptr[0]->ptr ) ;

  page_bitmap_ptr[1] = (imageBitmap*) d_data->page_bitmap_ptr[1] ;
  argprintf ( "page_bitmap_ptr[1]: %p, page_bitmap_ptr[1]->ptr: %p", page_bitmap_ptr[1], page_bitmap_ptr[1]->ptr ) ;

  page_bitmap_ptr[2] = (imageBitmap*) d_data->page_bitmap_ptr[2] ;
  argprintf ( "page_bitmap_ptr[2]: %p, page_bitmap_ptr[2]->ptr: %p", page_bitmap_ptr[2], page_bitmap_ptr[2]->ptr ) ;

  page_bitmap_ptr[3] = (imageBitmap*) d_data->page_bitmap_ptr[3] ;
  argprintf ( "page_bitmap_ptr[3]: %p, page_bitmap_ptr[3]->ptr: %p", page_bitmap_ptr[3], page_bitmap_ptr[3]->ptr ) ;

  imageBitmap *buffer_bitmap_ptr = (imageBitmap*) d_data->buffer_bitmap_ptr ;
  argprintf ( "buffer_bitmap_ptr: %p, buffer_bitmap_ptr->ptr: %p", buffer_bitmap_ptr, buffer_bitmap_ptr->ptr ) ;

  dbgprintf ( "dumping flash" ) ;

  starttime = time ( NULL ) ;

  vflash = vflash_state () ;
  argprintf ( "vflash: %s", vflash ? "on" : "off" ) ;

  if ( vflash )
  {
    device_id = DUMP_FLASH_NOR_DEVID ;
  }
  else
  {
    device_id = DUMP_FLASH_NAND_DEVID ;
  }

  /* Patching LV1 hypervisor to unmask bootldr by HV */
  if ( ( ret = patchUnmaskBootldr () ) != 0 )
  {
    errprintf ( "LV1 unmask bootldr returned: 0x%x (%d)", ret, ret ) ;
    return ( -1 ) ;
  }
  else
  {
    mask = 1 ;
  }

  dumpingSendMemoryLed ( DUMP_LED_NONE, d_data ) ;

  dumpingSendIoLed ( DUMP_LED_FLASH, d_data ) ;

  if ( ( ret = storage_open ( device_id, &device_handle ) ) != 0 )
  {
    errprintf ( "storage_open: 0x%08x", ret ) ;
    goto done ;
  }

  if ( ( ret = storage_get_device_info ( device_id, &device_info ) ) != 0 )
  {
    errprintf ( "storage_get_device_info: 0x%08x", ret ) ;
    goto done ;
  }

  sector_count = device_info.capacity ;
  size = ( sector_count * sector_size ) ;

  argprintf ( "FLASH res1: 0x%08x, res1: 0x%08x, res1: 0x%08x, res1: 0x%08x", device_info.res1[0], device_info.res1[1], device_info.res1[2], device_info.res1[3] ) ;
  argprintf ( "FLASH res1: 0x%08x, res1: 0x%08x, res1: 0x%08x, res1: 0x%08x", device_info.res1[4], device_info.res1[5], device_info.res1[6], device_info.res1[7] ) ;
  argprintf ( "FLASH vendor id: 0x%08x, device id: 0x%08x", device_info.vendor_id, device_info.device_id ) ;
  argprintf ( "FLASH capacity: 0x%016"PRIx64", sector count: 0x%x", device_info.capacity, sector_count ) ;
  argprintf ( "FLASH media count: 0x%08x", device_info.media_count ) ;
  argprintf ( "FLASH res2: 0x%08x, res2: 0x%08x, res2: 0x%08x, res2: 0x%08x", device_info.res2[0], device_info.res2[1], device_info.res2[2], device_info.res2[3] ) ;
  argprintf ( "FLASH dump size: 0x%016"PRIx64, size ) ;

  dumpingSendIoLed ( DUMP_LED_DISK, d_data ) ;

  aio_data.buffered = fileAioBuffered ( size ) ;

  if ( ( ret = fileAioOpen ( store, filename, &aio_data ) ) != 0 )
  {
    errprintf ( "opening dump file: %s", filename ) ;
    goto done ;
  }

  /* allocate user memory buffer */
  usr_buffer_ptr = malloc ( PAGE_SIZE ) ;

  /* clear buffer */
  if ( usr_buffer_ptr != NULL )
  {
    memset ( usr_buffer_ptr, 0, PAGE_SIZE ) ;
  }
  else
  {
    ret = -1 ;
    goto done ;
  }

  if ( buffer_bitmap_ptr->ptr != NULL )
  {
    memset ( buffer_bitmap_ptr->ptr, 0, 512 * PAGE_SIZE ) ;
  }
  else
  {
    ret = -1 ;
    goto done ;
  }

  percent_pages = ( (u32) size / PAGE_SIZE ) / 100 ;

  argprintf ( "rows: %d, frames: %d, pages/percent: %d",
              ( (u32) size / PAGE_SIZE ),
              ( (u32) size / PAGE_SIZE ) / 512,
              percent_pages ) ;

  /* set row 0 */
  d_data->row = 1 ;
  *d_data->action = 0 ;

  argprintf ( "FLASH: dumping sectors 0x%08x - 0x%08x, sector_count: 0x%x", start_sector, start_sector + sector_count, sector_count ) ;

  loopstart = __mftb () ;

  while ( sector_count >= NSECTORS )
  {
    if ( *d_data->exitapp == 0 || *d_data->action == -1 )
    {
      goto done ;
    }

    if ( percent_counter == percent_pages )
    {
      progress_counter++ ;
      percent_counter = 0 ;
      dumpingSendProgressLed ( progress_counter, d_data ) ;
    }
    else
    {
      percent_counter++ ;
    }

    copystart = __mftb () ;

    dumpingSendIoLed ( DUMP_LED_FLASH, d_data ) ;

    if ( ( ret = storage_read ( device_handle, 0, start_sector, NSECTORS, usr_buffer_ptr, &sectors_read, device_flags ) ) != 0 )
    {
      errprintf ( "storage_read: 0x%08x", ret ) ;
      goto done ;
    }

    if ( sectors_read != NSECTORS )
    {
      errprintf ( "bad read count: 0x%08x", sectors_read ) ;
      goto done ;
    }

    dumpingSendIoLed ( DUMP_LED_MEM, d_data ) ;

    memcpy ( page_bitmap_ptr[current_buffer]->ptr + current_page, usr_buffer_ptr, PAGE_SIZE ) ;

    copydone = __mftb () ;

    copytotal += ( copydone - copystart ) ;

    /* copy data into buffer */
    if ( page_counter == 15 )
    {
      videoutilImageCopy ( page_bitmap_ptr[current_buffer]->offset,
                           page_bitmap_ptr[current_buffer]->pitch,
                           ( buffer_bitmap_ptr->offset + ( ( d_data->row - 15 ) * PAGE_SIZE ) ),
                           buffer_bitmap_ptr->pitch,
                           current_page,
                           16,
                           d_data->vu_data ) ;

      /* send buffer update event */
      dumpingSendDumpUpdate ( d_data ) ;
      page_counter = 0 ;
      current_page = 0 ;
      current_buffer += current_buffer ;
      if ( current_buffer > 3 )
      {
        current_buffer = 0 ;
      }
      memset ( page_bitmap_ptr[current_buffer]->ptr, 0, 16 * PAGE_SIZE ) ;
    }
    else
    {
      page_counter++ ;
      current_page += PAGE_SIZE ;
    }

    /* buffer row counter */
    if ( d_data->row == buffer_bitmap_ptr->height )
    {
      d_data->row = 1 ;
    }
    else
    {
      d_data->row++ ;
    }

    dumpingSendIoLed ( DUMP_LED_DISK, d_data ) ;

    writestart = __mftb () ;

    if ( ( ret = fileAioWrite ( usr_buffer_ptr, PAGE_SIZE, &aio_data ) ) < 0 )
    {
      errprintf("write failed (0x%08x)", ret);
      goto done;
    }
    else
    {
      file_counter += PAGE_SIZE ;
    }

    writedone = __mftb () ;

    memset ( usr_buffer_ptr, 0, PAGE_SIZE ) ;

    dumpingSendIoLed ( DUMP_LED_NONE, d_data ) ;

    writetotal += ( writedone - writestart ) ;

    start_sector += NSECTORS ;
    sector_count -= NSECTORS ;

    count++ ;
  }

  dumpingSendDumpUpdate ( d_data ) ;
  argprintf ( "remaining sector_count: 0x%x", sector_count ) ;

  while ( sector_count )
  {
    memset ( usr_buffer_ptr, 0, PAGE_SIZE ) ;

    if ( *d_data->exitapp == 0 || *d_data->action == -1 )
    {
      goto done ;
    }

    if ( percent_counter == percent_pages )
    {
      progress_counter++ ;
      percent_counter = 0 ;
      dumpingSendProgressLed ( progress_counter, d_data ) ;
    }
    else
    {
      percent_counter++ ;
    }

    copystart = __mftb () ;

    dumpingSendIoLed ( DUMP_LED_FLASH, d_data ) ;

    if ( ( ret = storage_read ( device_handle, 0, start_sector, 1, usr_buffer_ptr, &sectors_read, device_flags ) ) != 0 )
    {
      errprintf ( "storage_read 0x%08x", ret ) ;
      goto done ;
    }

    if ( sectors_read != 1 )
    {
      errprintf ( "bad read count: 0x%08x", sectors_read ) ;
      goto done ;
    }

    dumpingSendIoLed ( DUMP_LED_MEM, d_data ) ;

    memcpy ( page_bitmap_ptr[current_buffer]->ptr + current_page, usr_buffer_ptr, PAGE_SIZE ) ;

    copydone = __mftb () ;

    copytotal += ( copydone - copystart ) ;

    /* copy data into buffer */
    if ( page_counter == 15 )
    {
      videoutilImageCopy ( page_bitmap_ptr[current_buffer]->offset,
                           page_bitmap_ptr[current_buffer]->pitch,
                           ( buffer_bitmap_ptr->offset + ( ( d_data->row - 15 ) * PAGE_SIZE ) ),
                           buffer_bitmap_ptr->pitch,
                           current_page,
                           16,
                           d_data->vu_data ) ;

      /* send buffer update event */
      dumpingSendDumpUpdate ( d_data ) ;
      page_counter = 0 ;
      current_page = 0 ;
      current_buffer += current_buffer ;
      if ( current_buffer > 3 )
      {
        current_buffer = 0 ;
      }
      memset ( page_bitmap_ptr[current_buffer]->ptr, 0, 16 * PAGE_SIZE ) ;
    }
    else
    {
      page_counter++ ;
      current_page += PAGE_SIZE ;
    }

    /* buffer row counter */
    if ( d_data->row == buffer_bitmap_ptr->height )
    {
      d_data->row = 1 ;
    }
    else
    {
      d_data->row++ ;
    }

    dumpingSendIoLed ( DUMP_LED_DISK, d_data ) ;

    writestart = __mftb () ;

    aio_data.buffered = 0 ;
    if ( ( ret = fileAioWrite ( usr_buffer_ptr, PAGE_SIZE, &aio_data ) ) < 0 )
    {
      errprintf("write failed (0x%08x)", ret);
      goto done;
    }
    else
    {
      file_counter += PAGE_SIZE ;
    }

    writedone = __mftb () ;

    memset ( usr_buffer_ptr, 0, PAGE_SIZE ) ;

    dumpingSendIoLed ( DUMP_LED_NONE, d_data ) ;

    writetotal += ( writedone - writestart ) ;

    start_sector += 1 ;
    sector_count -= 1 ;

    count++ ;
  }

  dumpingSendDumpUpdate ( d_data ) ;

  dumpingSendProgressLed ( DUMP_LED_DONE, d_data ) ;
  dumpingSendMemoryLed ( DUMP_LED_NONE, d_data ) ;
  dumpingSendIoLed ( DUMP_LED_NONE, d_data ) ;

  loopdone = __mftb () ;

  /* disable updates */
  d_data->row = -1 ;

  ret = 0 ;

done:
  fileAioClose ( &aio_data ) ;

  if ( ( ret = storage_close ( device_handle ) ) != 0 )
  {
    errprintf ( "storage_close 0x%08x", ret ) ;
  }

  donetime = time ( NULL ) ;

  sectime = difftime ( donetime, starttime ) ;

  looptotal = ( loopdone - loopstart ) ;
  loopavg = ( looptotal / count ) ;
  copyavg = copytotal / count ;
  writeavg = writetotal / count ;

  argprintf ( "%s dump done, wrote: %d bytes, seconds: %g, bytes/sec: %g", filename, file_counter, sectime, ((double)file_counter) / sectime ) ;
  argprintf ( "loop: %"PRId64":%"PRId64", copy: %"PRId64":%"PRId64", write: %"PRId64":%"PRId64, looptotal, loopavg, copytotal, copyavg, writetotal, writeavg );

  if ( usr_buffer_ptr )
  {
    free ( usr_buffer_ptr ) ;
  }

  dbgprintf ( "dumped flash" ) ;

  /* Patching LV1 hypervisor to mask bootldr by HV */
  if ( mask == 1 )
  {
    if ( ( ret = patchMaskBootldr () ) != 0 )
    {
      errprintf ( "LV1 mask returned: 0x%x (%d)", ret, ret ) ;
    }
  }

  return ( ret ) ;
 
#undef NSECTORS
}

