#line 1 "source/copy_memory.c"
/*
 * copy_memory.c
 *
 * Copyright (C) an0nym0us
 *
 * This software is distributed under the terms of the GNU General Public
 * License ("GPL") version 3, as published by the Free Software Foundation.
 *
 */

/*
 * includes
 */

#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <inttypes.h>
#include <time.h>

#include <ppu_intrinsics.h>

#include <sys/time.h>

#include "common.h"
#include "debug.h"
#include "patchutil.h"
#include "hexdump.h"

#include "file.h"
#include "map_memory.h"
#include "dumping.h"


/*
 * defines
 */

#ifndef MAX_BUFFERS
#define MAX_BUFFERS       4
#endif


/*
 * globals
 */

extern fwVersions fw_data ;
extern u32 log_fd ;
extern char string[MAX_STRING] ;


/*
 * functions
 */

/* copy memory */
s32
copy_memory ( s32 type, s32 store, u64 offset, u64 size, const char *filename, dumpData *d_data )
{
  aioData aio_data ;
  u8 *usr_buffer_ptr = NULL ;
  s32 ret = -1 ;
  u8 current_buffer = 0 ;

  u32 count = 0 ;
  s32 page_counter = 0 ;
  s32 file_counter = 0 ;
  s32 percent_pages = 0 ;
  s32 percent_counter = 0 ;
  s32 progress_counter = 1 ;
  s32 current_page = 0 ;

  time_t starttime = 0 ;
  time_t donetime = 0 ;
  double sectime = 0 ;
  u64 loopstart = 0 ;
  u64 loopdone = 0 ;
  u64 looptotal = 0 ;
  u64 loopavg = 0 ;
  u64 copystart = 0 ;
  u64 copydone = 0 ;
  u64 copytotal = 0 ;
  u64 copyavg = 0 ;
  u64 writestart = 0 ;
  u64 writedone = 0 ;
  u64 writetotal = 0 ;
  u64 writeavg = 0 ;

  u64 base = SYSCALL_HV_BASE ;
  u64 addr = 0ULL ;
  u64 off = 0ULL ;
  u64 mmap_lpar_addr = 0ULL ;

  imageBitmap *page_bitmap_ptr[MAX_BUFFERS] = { NULL, NULL } ;

  page_bitmap_ptr[0] = (imageBitmap*) d_data->page_bitmap_ptr[0] ;
  argprintf ( "page_bitmap_ptr[0]: %p, page_bitmap_ptr[0]->ptr: %p", page_bitmap_ptr[0], page_bitmap_ptr[0]->ptr ) ;

  page_bitmap_ptr[1] = (imageBitmap*) d_data->page_bitmap_ptr[1] ;
  argprintf ( "page_bitmap_ptr[1]: %p, page_bitmap_ptr[1]->ptr: %p", page_bitmap_ptr[1], page_bitmap_ptr[1]->ptr ) ;

  page_bitmap_ptr[2] = (imageBitmap*) d_data->page_bitmap_ptr[2] ;
  argprintf ( "page_bitmap_ptr[2]: %p, page_bitmap_ptr[2]->ptr: %p", page_bitmap_ptr[2], page_bitmap_ptr[2]->ptr ) ;

  page_bitmap_ptr[3] = (imageBitmap*) d_data->page_bitmap_ptr[3] ;
  argprintf ( "page_bitmap_ptr[3]: %p, page_bitmap_ptr[3]->ptr: %p", page_bitmap_ptr[3], page_bitmap_ptr[3]->ptr ) ;

  imageBitmap *buffer_bitmap_ptr = (imageBitmap*) d_data->buffer_bitmap_ptr ;
  argprintf ( "buffer_bitmap_ptr: %p, buffer_bitmap_ptr->ptr: %p", buffer_bitmap_ptr, buffer_bitmap_ptr->ptr ) ;

  dbgprintf ( "copying memory" ) ;

  starttime = time ( NULL ) ;

  dumpingSendMemoryLed ( DUMP_LED_NONE, d_data ) ;

  dumpingSendIoLed ( DUMP_LED_DISK, d_data ) ;

  aio_data.buffered = fileAioBuffered ( size ) ;

  if ( ( ret = fileAioOpen ( store, filename, &aio_data ) ) != 0 )
  {
    errprintf ( "opening dump file: %s", filename ) ;
    goto done ;
  }

  /* allocate user memory buffer */
  usr_buffer_ptr = malloc ( PAGE_SIZE ) ;

  /* clear buffer */
  if ( usr_buffer_ptr != NULL )
  {
    memset ( usr_buffer_ptr, 0, PAGE_SIZE ) ;
  }
  else
  {
    ret = -1 ;
    goto done ;
  }

  if ( buffer_bitmap_ptr->ptr != NULL )
  {
    memset ( buffer_bitmap_ptr->ptr, 0, 512 * PAGE_SIZE ) ;
  }
  else
  {
    ret = -1 ;
    goto done ;
  }

  percent_pages = ( (u32) size / PAGE_SIZE ) / 100 ;

  argprintf ( "rows: %d, frames: %d, pages/percent: %d",
              ( (u32) size / PAGE_SIZE ),
              ( (u32) size / PAGE_SIZE ) / 512,
              percent_pages ) ;

  /* set row 0 */
  d_data->row = 1 ;
  *d_data->action = 0 ;

  argprintf ( "copying offset 0x%016" PRIx64 " size 0x%016" PRIx64" to %s", offset, size, filename);

  loopstart = __mftb () ;

  dumpingSendMemoryLed ( type, d_data ) ;

  loopstart = __mftb () ;

  for ( off = offset;off < offset + size;off += PAGE_SIZE )
  {
    if ( *d_data->exitapp == 0 || *d_data->action == -1 )
    {
      goto done ;
    }

    addr = off ;

    if ( percent_counter == percent_pages )
    {
      progress_counter++ ;
      percent_counter = 0 ;
      dumpingSendProgressLed ( progress_counter, d_data ) ;
    }
    else
    {
      percent_counter++ ;
    }

    copystart = __mftb () ;

    dumpingSendIoLed ( DUMP_LED_MEM, d_data ) ;

    if ( type == TYPE_LV1 )
    {
      if ( ( ret = map_memory ( addr, SYSCALL_HV_PAGE_SIZE, &mmap_lpar_addr, base ) ) != 0 )
      {
        errprintf ( "error mapping lv1 into lv2 memory: 0x%x (%d)", ret, ret ) ;
        goto done ;
      }
      else
      {
        addr = base ;
      }
    }

    if ( ( ret = syscall_copy_to_user ( (u64)addr, (u64)usr_buffer_ptr, PAGE_SIZE ) ) < 0 )
    {
      argprintf ( "syscall_copy_to_user returned: 0x%x, offset: 0x%016"PRIx64, ret, addr ) ;
      goto done ;
    }

    memcpy ( page_bitmap_ptr[current_buffer]->ptr + current_page, usr_buffer_ptr, PAGE_SIZE ) ;

    if ( type == TYPE_LV1 )
    {
      unmap_memory ( &mmap_lpar_addr ) ;
    }

    copydone = __mftb () ;

    copytotal += ( copydone - copystart ) ;

    /* copy data into buffer */
    if ( page_counter == 15 )
    {
      videoutilImageCopy ( page_bitmap_ptr[current_buffer]->offset,
                           page_bitmap_ptr[current_buffer]->pitch,
                           ( buffer_bitmap_ptr->offset + ( ( d_data->row - 15 ) * PAGE_SIZE ) ),
                           buffer_bitmap_ptr->pitch,
                           current_page,
                           16,
                           d_data->vu_data ) ;

      /* send buffer update event */
      dumpingSendDumpUpdate ( d_data ) ;
      page_counter = 0 ;
      current_page = 0 ;
      current_buffer += current_buffer ;
      if ( current_buffer > 3 )
      {
        current_buffer = 0 ;
      }
      memset ( page_bitmap_ptr[current_buffer]->ptr, 0, 16 * PAGE_SIZE ) ;
    }
    else
    {
      page_counter++ ;
      current_page += PAGE_SIZE ;
    }

    /* buffer row counter */
    if ( d_data->row == buffer_bitmap_ptr->height )
    {
      d_data->row = 1 ;
    }
    else
    {
      d_data->row++ ;
    }

    dumpingSendIoLed ( DUMP_LED_DISK, d_data ) ;

    writestart = __mftb () ;

    if ( ( ret = fileAioWrite ( usr_buffer_ptr, PAGE_SIZE, &aio_data ) ) < 0 )
    {
      errprintf("write failed (0x%08x)", ret);
      goto done;
    }
    else
    {
      file_counter += PAGE_SIZE ;
    }

    writedone = __mftb () ;

    memset ( usr_buffer_ptr, 0, PAGE_SIZE ) ;

    dumpingSendIoLed ( DUMP_LED_NONE, d_data ) ;

    writetotal += ( writedone - writestart ) ;

    count++ ;
  }

  dumpingSendDumpUpdate ( d_data ) ;

  dumpingSendProgressLed ( DUMP_LED_DONE, d_data ) ;
  dumpingSendMemoryLed ( DUMP_LED_NONE, d_data ) ;
  dumpingSendIoLed ( DUMP_LED_NONE, d_data ) ;

  loopdone = __mftb () ;

  /* disable updates */
  d_data->row = -1 ;

  ret = 0 ;

done:
  fileAioClose ( &aio_data ) ;

  donetime = time ( NULL ) ;

  sectime = difftime ( donetime, starttime ) ;

  looptotal = ( loopdone - loopstart ) ;
  loopavg = ( looptotal / count ) ;
  copyavg = copytotal / count ;
  writeavg = writetotal / count ;

  argprintf ( "%s copy done, wrote: %d bytes, seconds: %g, bytes/sec: %g", filename, file_counter, sectime, ((double)file_counter) / sectime ) ;
  argprintf ( "total:avg, loop: %"PRId64":%"PRId64", copy: %"PRId64":%"PRId64", write: %"PRId64":%"PRId64, looptotal, loopavg, copytotal, copyavg, writetotal, writeavg );

  if ( usr_buffer_ptr )
  {
    free ( usr_buffer_ptr ) ;
  }

  dbgprintf ( "copied memory" ) ;

  return ( ret ) ;
}

