#include "common.h"
#include "peek_poke.h"
#include "hvcall.h"
#include "mm.h"

#include <psl1ght/lv2.h>
#include <stdio.h>
#include <malloc.h>
#include <string.h>
#include <assert.h>
#include <unistd.h>
#include <sysutil/video.h>
#include <rsx/gcm.h>
#include <rsx/reality.h>
#include <io/pad.h>
#include <sys/stat.h>
#include <dirent.h>

#include "sconsole.h"

u64 mmap_lpar_addr = 0;
int debug_sock = -1;
char debug_buffer[1024];

#define PATCH_FLAG_EXEC		1
#define PATCH_FLAG_BACKUP	2
#define PATCH_FLAG_DEREF	4
#define PATCH_FLAG_LV1		8

static const char *search_dirs[13] = {
	"/dev_hdd0/game/LV2000000/USRDIR",

	"/dev_usb000/lv2",
	"/dev_usb001/lv2",
	"/dev_usb002/lv2",
	"/dev_usb003/lv2",
	"/dev_usb004/lv2",
	"/dev_usb005/lv2",
	"/dev_usb006/lv2",
	"/dev_usb007/lv2",

	"/dev_cf/lv2",
	"/dev_sd/lv2",
	"/dev_ms/lv2",

	NULL
};

unsigned char *read_file(FILE * f, size_t * sz)
{
	if (!f)
		return NULL;

	fseek(f, 0, SEEK_END);
	*sz = ftell(f);
	fseek(f, 0, SEEK_SET);

	unsigned char *userlandBuffer = malloc(*sz);
	if (!userlandBuffer)
		return NULL;

	fread(userlandBuffer, 1, *sz, f);
	fclose(f);

	if (*((u32 *) userlandBuffer) == 0) {
		free(userlandBuffer);
		userlandBuffer = NULL;
	}

	return userlandBuffer;
}

char filename_buf[256];
char last_dir[256];

FILE *search_file(char *filename)
{
	FILE *result;
	const char **search_dir = search_dirs;
	u32 test_value;

	if (*filename == '/')
	{
		strcpy (last_dir, "/");
		return fopen(filename, "r");
	}

	while (*search_dir) {
		strcpy(filename_buf, *search_dir);
		strcpy(last_dir, *search_dir);
		strcat(filename_buf, filename);
		result = fopen(filename_buf, "r");
		if (result) {
			test_value = 0;
			fread(&test_value, sizeof(u32), 1, result);
			fseek(result, 0, SEEK_SET);
			if (test_value == 0) {
				fclose(result);
				result = NULL;
			}
		}
		if (result)
			return result;
		else
			search_dir++;
	}

	return NULL;
}

int map_lv1()
{
	int result;

	if (!mmap_lpar_addr)
	{
		result =
		    lv1_undocumented_function_114(0, 0xC, HV_SIZE, &mmap_lpar_addr);
		if (result != 0) {
			PRINTF("Error code %d calling lv1_undocumented_function_114\n",
			       result);
			return 0;
		}
	}

	result =
	    mm_map_lpar_memory_region(mmap_lpar_addr, HV_BASE, HV_SIZE, 0xC, 0);
	if (result) {
		PRINTF("Error code %d calling mm_map_lpar_memory_region\n",
		       result);
		return 0;
	}

	return 1;
}

void unmap_lv1()
{
	if (mmap_lpar_addr != 0)
		lv1_undocumented_function_115(mmap_lpar_addr);
}

void patch_lv2_protection()
{
	// changes protected area of lv2 to first byte only
	lv1_poke(0x363a78, 0x0000000000000001ULL);
	lv1_poke(0x363a80, 0xe0d251b556c59f05ULL);
	lv1_poke(0x363a88, 0xc232fcad552c80d7ULL);
	lv1_poke(0x363a90, 0x65140cd200000000ULL);
}

typedef struct {
	int height;
	int width;
	uint32_t *ptr;
	// Internal stuff
	uint32_t offset;
} buffer;

gcmContextData *context;
VideoResolution res;
int currentBuffer = 0;
buffer *buffers[2];

void waitFlip()
{
	// Block the PPU thread untill the previous flip operation has finished.
	while (gcmGetFlipStatus() != 0)
		usleep(200);
	gcmResetFlipStatus();
}

void flip(s32 buffer)
{
	assert(gcmSetFlip(context, buffer) == 0);
	realityFlushBuffer(context);
	gcmSetWaitFlip(context);
}

void makeBuffer(int id, int size)
{
	buffer *buf = malloc(sizeof(buffer));
	buf->ptr = rsxMemAlign(16, size);
	assert(buf->ptr != NULL);

	assert(realityAddressToOffset(buf->ptr, &buf->offset) == 0);
	assert(gcmSetDisplayBuffer
	       (id, buf->offset, res.width * 4, res.width, res.height) == 0);

	buf->width = res.width;
	buf->height = res.height;
	buffers[id] = buf;
}

void init_screen()
{
	void *host_addr = memalign(1024 * 1024, 1024 * 1024);
	assert(host_addr != NULL);

	context = realityInit(0x10000, 1024 * 1024, host_addr);
	assert(context != NULL);

	VideoState state;
	assert(videoGetState(0, 0, &state) == 0);
	assert(state.state == 0);

	assert(videoGetResolution(state.displayMode.resolution, &res) == 0);

	VideoConfiguration vconfig;
	memset(&vconfig, 0, sizeof(VideoConfiguration));
	vconfig.resolution = state.displayMode.resolution;
	vconfig.format = VIDEO_BUFFER_FORMAT_XRGB;
	vconfig.pitch = res.width * 4;

	assert(videoConfigure(0, &vconfig, NULL, 0) == 0);
	assert(videoGetState(0, 0, &state) == 0);

	s32 buffer_size = 4 * res.width * res.height;

	gcmSetFlipMode(GCM_FLIP_VSYNC);
	makeBuffer(0, buffer_size);
	makeBuffer(1, buffer_size);

	gcmResetFlipStatus();
	flip(1);
}

void go(void *addr)
{
	u64 syscall11_ptr = lv2_peek(SYSCALL_PTR(11));
	u64 old_syscall11 = lv2_peek(syscall11_ptr);
	lv2_poke(syscall11_ptr, (u64) addr);
	Lv2Syscall0(11);
	lv2_poke(syscall11_ptr, old_syscall11);
}

void install_lv2_memcpy()
{
	PRINTF("installing memcpy...\n");
	/* install memcpy */
	lv2_poke(NEW_POKE_SYSCALL_ADDR, 0x4800000428250000ULL);
	lv2_poke(NEW_POKE_SYSCALL_ADDR + 8, 0x4182001438a5ffffULL);
	lv2_poke(NEW_POKE_SYSCALL_ADDR + 16, 0x7cc428ae7cc329aeULL);
	lv2_poke(NEW_POKE_SYSCALL_ADDR + 24, 0x4bffffec4e800020ULL);
}

void remove_lv2_memcpy()
{
	PRINTF("uninstalling memcpy...\n");
	/* restore syscall */
	remove_new_poke();
	lv2_poke(NEW_POKE_SYSCALL_ADDR + 16, 0xebc2fe287c7f1b78);
	lv2_poke(NEW_POKE_SYSCALL_ADDR + 24, 0x3860032dfba100e8);
}

inline static void lv2_memcpy(void *to, const void *from, size_t sz)
{
	Lv2Syscall3(NEW_POKE_SYSCALL, (unsigned long long)to,
		    (unsigned long long)
		    from, sz);
}

void exec_cmd(const char *patch_dir, char *buf, int *patches, int *payloads);
void apply_patches(const char *patch_dir, FILE *patch, int *patches, int *payloads);

void debug_mode()
{
	char buf[512], *ptr;
#if DEBUG
#else
	debug_wait_for_client ();
#endif
	while (1)
	{
		int patches = 0, payloads = 0;
		if (write (debug_sock, "(dbg) ", 6) <= 0)
			break;
		ptr = buf;
		while (1)
		{
			if (read (debug_sock, ptr, 1) != 1)
		  		return;
			if (write (debug_sock, ptr, 1) != 1)
				return;
			if (*ptr == '\n' || *ptr == '\r')
				break;
			else
				ptr++;
		}
		*++ptr = 0;

		while ((ptr = strchr (buf, '\b')))
			memmove (ptr, ptr + 1, strlen (ptr));
		if (*buf)
			exec_cmd (*search_dirs, buf, &patches, &payloads);
		if (patches)
			PRINTF ("[patch has been applied]\n");
		if (payloads)
			PRINTF ("[payload has been loaded]\n");
	}
	close(debug_sock);
	debug_sock = -1;
}

void exec_cmd(const char *patch_dir, char *buf, int *patches, int *payloads)
{
	FILE *payload;
	void *addr, *backup_addr;
	char *ptr, *ptr2, flags = 0;
	unsigned long long value;
	static void* last_haystack;
	static unsigned long long last_needle;
	static int last_size = 0;
	int i;

	/*
	 * [*]address: [@[x][b][1]] { payload_name | poke32 | poke64 | "go" | alloc <bytes> }
	 *
	 * 472461: xb payload.bin
	 * 28ca70: 37719461
	 * 7f918a: 16380059372ab00a
	 */

	flags = 0;
	ptr = strchr(buf, '#');
	if (ptr)
		*ptr = 0;
	ptr = buf;
	while (*ptr == ' ' || *ptr == '\t')
		ptr++;
	if (*ptr == '\n' || *ptr == '\r' || !*ptr)
		return;
	PRINTF ("exec_cmd: %s;\n", buf);
	if (*ptr == '*')
	{
		flags |= PATCH_FLAG_DEREF;
		ptr++;
	}
#if 0
	if (!strchr("0123456789abcdefABCDEF", *ptr))
	{
		PRINTF("Some strange line: %s.\n", ptr);
		return;
	}
#endif
	ptr2 = ptr;
	addr = (void *)strtoull(ptr, &ptr, 16);
	if (*ptr == ':')
		ptr++;
	else if (ptr != ptr2)
	{
		PRINTF ("Syntax error.\n");
		return;
	}
	else
		addr = 0;
	while (*ptr == ' ' || *ptr == '\t')
		ptr++;
	if (*ptr == '@') {
		ptr++;
		if (*ptr == 'x') {
			flags |= PATCH_FLAG_EXEC;
			ptr++;
		}
		if (*ptr == 'b') {
			flags |= PATCH_FLAG_BACKUP;
			ptr++;
		}
		if (*ptr == '1') {
			flags |= PATCH_FLAG_LV1;
			ptr++;
		}
		while (*ptr == ' ' || *ptr == '\t')
			ptr++;
	}
	if (ptr[0] == 'g' && ptr[1] == 'o') {
		if (flags & PATCH_FLAG_DEREF)
			addr = (void *) lv2_peek((u64) addr);
		else
			addr += 0x8000000000000000ULL;
		if (!addr)
		{
			PRINTF ("Null pointer exception.\n");
			return;
		}
		go (addr);
	} else if (ptr[0] == '.' && (ptr[1] == ' ' || ptr[1] == '\t')) {
		while (*ptr && strchr(" \t", *ptr))
			ptr++;
		payload = fopen(ptr, "r");
		if (!payload)
		{
			PRINTF("Cannot open file.\n");
			return;
		}

		apply_patches(patch_dir, payload, patches, payloads);
		fclose (payload);
	} else if (ptr[0] == 'l' && ptr[1] == 'v' &&
	           ptr[2] == '1' && ptr[3] == 'e' &&
		   ptr[4] == 'n') {
		map_lv1();
	} else if (ptr[0] == 'l' && ptr[1] == 'v' &&
	           ptr[2] == '1' && ptr[3] == 'd' &&
		   ptr[4] == 'i') {
	  	unmap_lv1();
	} else if (ptr[0] == 'p' && ptr[1] == 'a' &&
	           ptr[2] == 'n' && ptr[3] == 'i' &&
		   ptr[4] == 'c') {
		lv2_poke(NEW_POKE_SYSCALL_ADDR, 0x38600001396000ff);
		lv2_poke(NEW_POKE_SYSCALL_ADDR + 8, 0x440000224e800020ULL);
		Lv2Syscall0(NEW_POKE_SYSCALL);
	} else if (ptr[0] == 's' && ptr[1] == 'e' && 
		   ptr[2] == 'a' && ptr[3] == 'r' &&
		   ptr[4] == 'c' && ptr[5] == 'h') {
		ptr += 6;
		while (*ptr && strchr(" \t", *ptr))
			ptr++;
		if (!*ptr)
		{
			PRINTF("Syntax error.\n");
			return;
		}
		ptr2 = ptr;
		value = strtoull(ptr, &ptr, 16);

		backup_addr = addr;

		last_size = ptr - ptr2;
		last_needle = value;

		if (ptr - ptr2 == 8)
		{
			for (addr = (void *) 0x8000000000000000ULL; addr < (void *) 0x8000000000780000; addr = (void *) ((u64) addr) + 4)
			{
				if ( ((lv2_peek ((u64) addr) >> 32) & (0xffffffffULL)) == value )
				{
					PRINTF("0x%08llX ound at 0x%lX.\n", value, ((u64) addr) - 0x8000000000000000);
					break;
				}
			}
			last_haystack = (void *) ((u64) addr) + 4;
			if (addr == (void *) 0x8000000000780000)
			{
				PRINTF("Not found.\n");
				return;
			}
		}
		else if (ptr - ptr2 == 16)
		{
			for (addr = (void *) 0x8000000000000000ULL; addr < (void *) 0x8000000000780000; addr = (void *) ((u64) addr) + 8)
			{
				if ( lv2_peek ((u64) addr) == value )
				{
					PRINTF("0x%16llX found at 0x%lX.\n", value, ((u64) addr) - 0x8000000000000000);
					break;
				}
			}
			last_haystack = (void *) ((u64) addr) + 8;
			if (addr == (void *) 0x8000000000780000)
			{
				PRINTF("Not found.\n");
				return;
			}
		}
		else
		{
			PRINTF("Invalid needle size: %ld.\n", ptr - ptr2);
			return;
		}

		if (backup_addr && (flags & PATCH_FLAG_DEREF))
		{
			lv2_poke ((u64) backup_addr, (u64) addr);
		}
	} else if (ptr[0] == 'n' && (ptr[1] == '\n' || ptr[1] == '\r' || ptr[1] == 0)) {
		backup_addr = addr;

		if (last_size == 8)
		{
			for (addr = last_haystack; addr < (void *) 0x8000000000780000; addr = (void *) ((u64) addr) + 4)
			{
				if ( ((lv2_peek ((u64) addr) >> 32) & (0xffffffffULL)) == last_needle )
				{
					PRINTF("0x%08llX ound at 0x%lX.\n", last_needle, ((u64) addr) - 0x8000000000000000);
					break;
				}
			}
			if (addr == (void *) 0x8000000000780000)
			{
				PRINTF("Not found.\n");
				return;
			}
		}
		else if (last_size == 16)
		{
			for (addr = last_haystack; addr < (void *) 0x8000000000780000; addr = (void *) ((u64) addr) + 8)
			{
				if ( lv2_peek ((u64) addr) == last_needle )
				{
					PRINTF("0x%16llX found at 0x%lX.\n", last_needle, ((u64) addr) - 0x8000000000000000);
					break;
				}
			}
			if (addr == (void *) 0x8000000000780000)
			{
				PRINTF("Not found.\n");
				return;
			}
		}
		else
		{
			PRINTF("Invalid needle size: %ld.\n", ptr - ptr2);
			return;
		}
	
		if (backup_addr && (flags & PATCH_FLAG_DEREF))
		{
			lv2_poke ((u64) backup_addr, (u64) addr);
		}
	} else if (ptr[0] == 'p' && (ptr[1] == ' ' || ptr[1] == '\t')) {
		ptr += 2;
		while (*ptr && strchr(" \t", *ptr))
			ptr++;

		backup_addr = addr;
		addr = (void *) strtoull(ptr, &ptr, 16);

		value = lv2_peek(((u64) addr) + 0x8000000000000000ULL);

		PRINTF ("%8lX: %16llX\n", (u64) addr, value);

		if (backup_addr && (flags & PATCH_FLAG_DEREF))
			lv2_poke((u64) backup_addr, value);
	} else if (ptr[0] == 'h' && (ptr[1] == ' ' || ptr[1] == '\t' || (ptr[1] == 'q' && (ptr[2] == ' ' || ptr[2] == '\t')))) {
		ptr2 = ptr;
		ptr += 2;
		while (*ptr && strchr(" \t", *ptr))
			ptr++;

		addr = (void *) strtoull(ptr, &ptr, 16);

		while (*ptr && strchr(" \t", *ptr))
			ptr++;

		backup_addr = (void *) strtoull(ptr, &ptr, 16);

		if (!backup_addr)
		{
			PRINTF ("h: Syntax error.\n");
			return;
		}

		if (backup_addr < addr)
			backup_addr = (void *) ((u64) backup_addr) + ((u64) addr);

		if (ptr2[1] == 'q')
			while (addr < backup_addr)
			{
				value = lv2_peek (((u64) addr) + 0x8000000000000000ULL);

				PRINTF ("%8lX: %16llX ", (u64) addr, value);
				for (i=0; i<8; i++)
				{
					if ((value >> (64 - 8)) >= ' ' && (value >> (64 - 8)) < 0x80)
						PRINTF("%c", (unsigned char) (value >> (64 - 8)));
					else
						PRINTF(".");
					value <<= 8;
				}
				PRINTF("\n");

				addr += 8;
			}
		else
			while (addr < backup_addr)
			{
				value = lv2_peek (((u64) addr) + 0x8000000000000000ULL);
				value >>= 32;
				value &= 0xffffffffULL;

				PRINTF ("%8lX: %08llX ", (u64) addr, value);
				for (i=0; i<4; i++)
				{
					if ((value >> (32 - 8)) >= ' ' && (value >> (32 - 8)) < 0x80)
						PRINTF("%c", (unsigned char) (value >> (32 - 8)));
					else
						PRINTF(".");
					value <<= 8;
				}
				PRINTF("\n");

				addr += 4;
			}
	} else if (ptr[0] == 'a' && ptr[1] == 'l' && 
		   ptr[2] == 'l' && ptr[3] == 'o' && 
		   ptr[4] == 'c') {
		ptr += 5;
		while (*ptr && strchr(" \t", *ptr))
			ptr++;
		if (!*ptr)
		{
			PRINTF("alloc: Syntax error.\n");
			return;
		}
		value = strtoull(ptr, &ptr, 16);

		addr += 0x8000000000000000ULL;
		ptr = (void *) addr;

		if (flags & PATCH_FLAG_LV1)
			* ((void **) ptr) = lv1_alloc(value, NULL);
		else
			* ((void **) ptr) = lv2_alloc(value, 0x27);
		
		if (! * ((void **) ptr))
		{
			PRINTF ("Alloc failed.\n");
			return;
		}

		(*patches) ++;
	} else if (!strchr("0123456789abcdefABCDEF", *ptr))
		do {
			ptr2 = strchr(ptr, '\n');
			if (ptr2)
				*ptr2 = 0;
			ptr2 = strchr(ptr, '\r');
			if (ptr2)
				*ptr2 = 0;
			ptr2 = strchr(ptr, ' ');
			if (ptr2)
				*ptr2 = 0;
			ptr2 = strchr(ptr, '\t');
			if (ptr2)
				*ptr2 = 0;

			if (ptr[0] != '/')
			{
				strcpy (filename_buf, patch_dir);
				strcat (filename_buf, "/");
				strcat (filename_buf, ptr);
			}
			else
				strcpy (filename_buf, ptr);

			payload = fopen(filename_buf, "r");

			if (!payload) {
				PRINTF
				    ("Cannot open file \"%s\".\n",
				     filename_buf);
				return;
			}

			PRINTF("reading payload...\n");
			size_t sz;
			unsigned char *payload_bin =
			    read_file(payload, &sz);

			backup_addr = 0;
			if (flags & PATCH_FLAG_LV1)
			{
				addr = lv1_alloc(sz, addr);
				if (!addr)
				{
					PRINTF("payload allocation failed.\n");
					break;
				}
			}
			else if (!addr || (flags & PATCH_FLAG_DEREF))
			{
				if (flags & PATCH_FLAG_DEREF)
				{
					backup_addr = addr + 0x8000000000000000ULL;
					flags &= ~PATCH_FLAG_BACKUP;
				}
				addr = lv2_alloc((sz + 7) & ~7, 0x27);
				if (!addr)
				{
					PRINTF("payload allocation failed.\n");
					break;
				}
			}
			else {
				addr += 0x8000000000000000ULL;
				if (flags & PATCH_FLAG_BACKUP)
					backup_addr =
					    lv2_alloc(sz, 0x27);
			}

			install_lv2_memcpy();

			if (flags & PATCH_FLAG_BACKUP) {
				/* backup */
				PRINTF
				    ("backing up the data...\n");
				lv2_memcpy(backup_addr, addr,
					   sz);
			}

			/* copy the payload */
			PRINTF("copying the payload...\n");
			lv2_memcpy((void *)
				   addr, payload_bin, sz);
			remove_lv2_memcpy();

			if (flags & PATCH_FLAG_EXEC) {
				PRINTF
				    ("Executing the payload at %lX...\n", (u64) addr);
				go(addr);
			}

			if (flags & PATCH_FLAG_BACKUP) {
				PRINTF
				    ("Restoring LV2 memory...\n");
				install_lv2_memcpy();
				lv2_memcpy(addr, backup_addr,
					   sz);
				remove_lv2_memcpy();
			}

			if (flags & PATCH_FLAG_DEREF)
			{
				lv2_poke ((u64) backup_addr, (u64) addr);
			}

			PRINTF("Done.\n");

			(*payloads) ++;
		} while (0);
	else {
		ptr2 = ptr;
		value = strtoull(ptr, &ptr, 16);

		if (flags & PATCH_FLAG_DEREF)
			addr = (void *) lv2_peek ((u64) addr);
		else
			addr += 0x8000000000000000ULL;

		if (!addr)
		{
			PRINTF ("Null pointer exception.\n");
			return;
		}

		(*patches) ++;

		if (ptr - ptr2 == 8) {
			lv2_poke32((u64) addr, value);
			PRINTF("poke32 %p %08llX\n",
			       (void *)addr, value);
		} else if (ptr - ptr2 == 16) {
			lv2_poke((u64) addr, value);
			PRINTF("poke64 %p %16llX\n",
			       (void *)addr, value);
		} else
			(*patches) --;
	}

}

void apply_patches(const char *patch_dir, FILE *patch, int *patches, int *payloads)
{
	char buf[512];
	while (!feof(patch)) {
		if (!fgets(buf, sizeof(buf), patch))
			break;
		if (!buf[0])
			break;

		exec_cmd (patch_dir, buf, patches, payloads);
	}
}

#define MAX_PATCH_FILES	10

s32 main(s32 argc, const char *argv[])
{
	char patch_dirs[MAX_PATCH_FILES][256];
	FILE* patch_files[MAX_PATCH_FILES];
	char patch_descriptions[MAX_PATCH_FILES][256] = { { 0, }, };

	int n_patch_files = 0;
	int current_patch_idx = 0;

#if DEBUG
	debug_wait_for_client();
#endif

	PRINTF("installing new poke syscall\n");
	install_new_poke();

	PRINTF("mapping lv1\n");
	if (!map_lv1()) {
		remove_new_poke();
		exit(0);
	} else {
		PRINTF("patching lv2 mem protection\n");
		patch_lv2_protection();

		PRINTF("removing new poke syscall\n");
		remove_new_poke();
	}

	PadInfo padinfo;
	PadData paddata;
	int i, j;

	FILE *patch = NULL;

	int patches = 0, payloads = 0;

	init_screen();
	ioPadInit(7);
	/*
	   Init the console: arguments (background color, font color, framebuffer, screen width, screen height)
	   sconsoleInit(int bgColor, int fgColor, int screenWidth, int screenHeight)
	 */
	sconsoleInit(FONT_COLOR_BLACK, FONT_COLOR_WHITE, res.width, res.height);
	char ts[1000];

	const char **search_dir;
	for (search_dir = search_dirs; *search_dir; search_dir++)
	{
		DIR *dir = opendir(*search_dir);
		if (!dir)
			continue;
		struct dirent *e;
		while ((e = readdir (dir)) && n_patch_files < MAX_PATCH_FILES)
		{
			if ((strstr (e->d_name, ".txt") || strstr (e->d_name, ".TXT")) &&
			   !(strstr (e->d_name, "readme") || strstr (e->d_name, "README")))
			{
			  	strcpy (filename_buf, *search_dir);
				strcat (filename_buf, "/");
				strcat (filename_buf, e->d_name);
				strcpy (patch_dirs[n_patch_files], *search_dir);
				patch_files[n_patch_files] = fopen (filename_buf, "r");

				if (patch_files[n_patch_files])
				{
					fgets (patch_descriptions[n_patch_files], 
						sizeof(patch_descriptions[n_patch_files]),
						patch_files[n_patch_files]);
					rewind (patch_files[n_patch_files]);
					n_patch_files++;
				}
			}
		}
		closedir(dir);
	}

	if (n_patch_files == 1)
		patch = patch_files[current_patch_idx];
	else while (!patch) {
		ioPadGetInfo(&padinfo);
		for (i = 0; i < MAX_PADS; i++) {
			if (padinfo.status[i]) {
				ioPadGetData(i, &paddata);
				if (paddata.BTN_CROSS)
				{
					if (current_patch_idx < n_patch_files)
						patch = patch_files[current_patch_idx];
				}
				else if (paddata.BTN_CIRCLE)
				{
					return 0;
				}
				else if (paddata.BTN_TRIANGLE)
				{
					waitFlip();
					memset (buffers[currentBuffer]->ptr, 0, res.width * res.height * sizeof(u32));
					print(50, 50, "DEBUG MODE", buffers[currentBuffer]->ptr);
					flip(currentBuffer);
					currentBuffer = !currentBuffer;
					debug_mode();
				}
				else if (paddata.BTN_UP)
				{
					if (current_patch_idx > 0)
						current_patch_idx --;
				}
				else if (paddata.BTN_DOWN)
				{
					if (current_patch_idx < n_patch_files - 1)
						current_patch_idx ++;
				}
			}
		}

		usleep(100000);

		waitFlip();

		//background
		for (i = 0; i < res.height; i++) {
			for (j = 0; j < res.width; j++)
				buffers[currentBuffer]->ptr[i * res.width + j] =
				    FONT_COLOR_BLACK;
		}

		if (! n_patch_files)
			print(50, 150, "ERROR: No patch files have been found.", buffers[currentBuffer]->ptr);

		for (i = 0; i < n_patch_files; i++)
		{
			snprintf (ts, 1000, "%c %.12s:%s", (current_patch_idx == i) ? '>' : ' ',
				patch_dirs[i], patch_descriptions[i]);
			print(50, 100 + i * 40, ts, buffers[currentBuffer]->ptr);
		}
		print (50, 100 + MAX_PATCH_FILES * 40, "Press X to confirm, O to exit.", buffers[currentBuffer]->ptr);

		flip(currentBuffer);
		currentBuffer = !currentBuffer;
	}

	if (patch) {
	  	apply_patches(patch_dirs[current_patch_idx], patch, &patches, &payloads);
		fclose(patch);
	}

	while (1) {
		ioPadGetInfo(&padinfo);
		for (i = 0; i < MAX_PADS; i++) {
			if (padinfo.status[i]) {
				ioPadGetData(i, &paddata);
				if (paddata.BTN_CROSS)
					return 0;
			}
		}

		waitFlip();

		//background
		for (i = 0; i < res.height; i++) {
			for (j = 0; j < res.width; j++)
				buffers[currentBuffer]->ptr[i * res.width + j] =
				    FONT_COLOR_BLACK;
		}

		//Let's do some printing   
		print(50, 50, "Hello from Russia!", buffers[currentBuffer]->ptr);
		sprintf(ts,
			"Installed %d payloads, %d patches.",
			payloads, patches);
		print(50, 150, ts, buffers[currentBuffer]->ptr);
		print(50, 250, last_dir, buffers[currentBuffer]->ptr);
		print(50, 450, "Press X to quit", buffers[currentBuffer]->ptr);

		flip(currentBuffer);
		currentBuffer = !currentBuffer;
	}

	PRINTF("done, exiting\n");
	return 0;
}
