#include <tiny3d.h>
#include <stdio.h>
#include <unistd.h>
#include <sysmodule/sysmodule.h>
#include <pngdec/loadpng.h>


#include "c4.h"
#include "gui.h"
#include "menu.h"
#include "pad.h"
#include "utils.h"

extern GuiFont * font;


//resource files to include
INC_FILE( aboutWindow_png );
INC_FILE( dialogue_box_png );
INC_FILE( progressbar_outline_png );
INC_FILE( progressbar_empty_png );
INC_FILE( progressbar_png );

INC_FILE( background_png );
INC_FILE( player1_point_png );
INC_FILE( player2_point_png );
INC_FILE( player3_point_png );
INC_FILE( player4_point_png );
INC_FILE( font_ttf );

INC_FILE( button_png );
INC_FILE( button_over_png );

INC_FILE( bg_music_ogg );
INC_FILE( button_over_wav );
INC_FILE( button_click_wav );
INC_FILE( button_click2_wav );
INC_FILE( button_click_pcm );

INC_FILE( boardSprite_png );
INC_FILE( redPiece_png );
INC_FILE( blackPiece_png );



//global gui elements
GuiImageData *bgImgData = NULL;
GuiImage *bgImage = NULL;
GuiWindow *mainWindow = NULL;
GuiImageData * pointer[4] = { NULL, NULL, NULL, NULL };
GuiFont *font = NULL;
GuiSound *bgMusic = NULL;

//stuff dealing with the gui thread and syncronization
bool exitRequested = false;
static bool haltgui = true;
static bool guihalted = true;
static bool guiThreadRunning = false;

static sys_mutex_t hGuiMutex;
static sys_mutex_t h2GuiMutex;
static sys_mutex_t wGuiMutex;
static sys_cond_t guiWaitCondition;
static sys_cond_t haltWaitCondition;
static sys_mutex_t exitRequestMutex;

static sys_ppu_thread_t guiThread;

//variables for handeling game settings and whatnot
enum {HUMAN = 0, COMPUTER = 1};

#define MAX_PLAYER_NAME_LENGTH  20
#define BOARD_MAX_ROWS          15//setting these too high makes the cpu take too damn long to calculate its move
#define BOARD_MAX_COLS          15
#define BOARD_MIN_ROWS          5
#define BOARD_MIN_COLS          5

#define GUI_TEXT_COLOR          0xf0f0f0ff //almost black

#define VIEWPORT_MIN            -100
#define VIEWPORT_MAX            0
static int viewportX = -40;
static int viewportY = -40;

enum
{
	T_ALTERNATE = 0,
	T_P1,
	T_P2,
	T_WINNER,
	T_LOSER,
	T_MAX
};
enum
{
	M_WAIT_CPU,
	M_WAIT_PLAYER,
	M_ANIMATE,
	M_CHECK_GAME_END,
	M_RESET_BOARD,
	M_WAIT_NEW_GAME
};
struct GameInfo
{
	u8 player[ 2 ];
	int level;
	char name[ 2 ][ MAX_PLAYER_NAME_LENGTH + 1 ];
	u8 turn;
	u8 numPlayers;
	u16 move;
	u16 p1Score;
	u16 p2Score;
	u16 ties;
	u8 first;//who goes first each game

	int width;
	int height;
	int numberToConnect;
};
GameInfo gameInfo;

u32 FrameTimer = 0;
void MenuRender();
static void UpdateGui( u64 arg )
{
	guiThreadRunning = true;
	bool halt;

	while( 1 )
	{
		//check if GuiHalt() has been called
		sys_mutex_lock( hGuiMutex, 0 );
		halt = haltgui;
		sys_mutex_unlock( hGuiMutex );
		if( halt )
		{
			while( 1 )
			{
				sys_ppu_thread_yield();
				//signal GuiHalt() that this thread has halted
				if( sys_mutex_trylock( h2GuiMutex ) )
					continue;

				sys_cond_signal( haltWaitCondition );
				sys_mutex_unlock( h2GuiMutex );

				//wait for ResumeGui() to be called
				if( sys_mutex_trylock( wGuiMutex ) )
					continue;
				guihalted = true;
				sys_cond_wait( guiWaitCondition, 0 );
				guihalted = false;
				sys_mutex_unlock( wGuiMutex );

				break;
			}
		}
		else
		{
			//check if exit() has been called
			sys_mutex_lock( exitRequestMutex, 0 );
			bool exit = exitRequested;
			sys_mutex_unlock( exitRequestMutex );
			if( exit )
			{
				//fade out
				for( int i = 0; i < 255; i += 15 )
				{
					MenuPrepareToDraw();
					mainWindow->Draw();
					MenuDrawRectangle( i, WINDOW_WIDTH, WINDOW_HEIGHT, 0, 0, 0, 1, 0, 1 );
					tiny3d_Flip();
				}
				break;
			}

			//read joypads
			PadRead();

			//update gui elements with user input
			for( int i = 0; i < 4; i++ )
				mainWindow->Update( &userInput[ i ] );

			//draw the window
			MenuRender();



		}
		sys_ppu_thread_yield();
	}
	guiThreadRunning = false;

	//you must call this, kthx
	sys_ppu_thread_exit( 0 );
}

void InitGuiThread()
{
	//thread parameters
	u64 thread_arg = 0x1337;
	u64 priority = 2000;
	size_t stack_size = 0x2000;
	const char *thread_name = "Gui Thread";

	//initialize mutex & wait condition
	mutex_init( &exitRequestMutex );
	mutex_init( &hGuiMutex );
	mutex_init( &h2GuiMutex );
	mutex_init( &wGuiMutex );
	cond_init( &guiWaitCondition, &wGuiMutex );
	cond_init( &haltWaitCondition, &h2GuiMutex );

	int s = sys_ppu_thread_create(	&guiThread, UpdateGui, thread_arg, priority, stack_size, THREAD_JOINABLE, (char *)thread_name );
	if( s )
	{
		printf("create gui thread returned %i.  exiting...\n", s );
		exit( 0 );
	}

}

void HaltGui()
{
	while( 1 )
	{
		sys_ppu_thread_yield();
		//signal gui thread to wait
		if( sys_mutex_trylock( hGuiMutex ) )
			continue;

		haltgui = true;
		sys_mutex_unlock( hGuiMutex );

		//wait for gui thread to halt
		if( sys_mutex_trylock( h2GuiMutex ) )
			continue;

		sys_cond_wait( haltWaitCondition, 0 );
		sys_mutex_unlock( h2GuiMutex );

		break;
	}
}

void ResumeGui()
{
	u16 wtf = 1;
	while( 1 )
	{
		if( !wtf++ )
		{
			printf("threads deadlocked\n");
			exit( 0 );
		}
		sys_ppu_thread_yield();
		//signal gui thread to wait
		if( sys_mutex_trylock( hGuiMutex ) )
			continue;
		haltgui = false;
		sys_mutex_unlock( hGuiMutex );

		if( guiThreadRunning && !guihalted && !exitRequested )
			continue;

		if( sys_mutex_trylock( wGuiMutex ) )
			continue;
		sys_cond_signal( guiWaitCondition );
		sys_mutex_unlock( wGuiMutex );

		break;
	}
}

void exiting()
{
	//printf("exiting()\n");
	sys_mutex_lock( exitRequestMutex, 0 );
	exitRequested = true;						//signal threads to end
	sys_mutex_unlock( exitRequestMutex );

	ResumeGui();                                //make sure the gui thread is not stuck in wait condition before trying to join

	u64 retval;
	int t = sys_ppu_thread_join( guiThread, &retval );
	if( t )
		printf("gui thread tried to join with return: %llX, sys_ppu_thread_join returned %d\n", (unsigned long long int)retval, t );

	//destroy gui mutexes & wait conditions
	sys_mutex_destroy( exitRequestMutex );
	sys_mutex_destroy( hGuiMutex );
	sys_mutex_destroy( h2GuiMutex );
	sys_mutex_destroy( wGuiMutex );
	sys_cond_destroy( guiWaitCondition );
	sys_cond_destroy( haltWaitCondition );

	//delete global gui elements
	if( bgImgData )
		delete bgImgData;
	if( bgImage )
		delete bgImage;
	if( font )
		delete font;
	if( bgMusic )
		delete bgMusic;
	if( mainWindow )
		delete mainWindow;

	for( int i = 0; i < 4; i++ )
	{
		if( pointer[ i ] )
			delete pointer[ i ];
	}

	SysUnloadModule( SYSMODULE_PNGDEC );		//unload PNG
	ioPadEnd();									//close pad driver
	GuiSound::UnInit();							//unload sound spu and whatnot

}

void ErrorPrompt( const char* message )
{
    WindowPrompt( "ERROR!", message, "Ok", NULL );
}

int WindowPrompt(const char *title, const char *msg, const char *btn1Label, const char *btn2Label)
{
    int choice = -1;

    GuiWindow promptWindow(448,288);
    promptWindow.SetAlignment(ALIGN_CENTRE | ALIGN_MIDDLE);
    promptWindow.SetPosition(0, -10);
	//GuiSound btnSoundOver(button_over_pcm, button_over_pcm_size, SOUND_PCM);
    GuiImageData btnOutline( button_png, button_png_size );
    GuiImageData btnOutlineOver( button_over_png, button_over_png_size );
    GuiTrigger trigA;
    trigA.SetSimpleTrigger(-1, BTN_CROSS_ );

	//button sounds
	GuiSound btnSndOver( button_over_wav, button_over_wav_size, SOUND_WAV );
	GuiSound btnSndClick2( button_click2_wav, button_click2_wav_size, SOUND_WAV );
	btnSndClick2.SetVolume( 50 );

    GuiImageData dialogBox( dialogue_box_png, dialogue_box_png_size );
    GuiImage dialogBoxImg(&dialogBox);

    GuiText titleTxt( font, title, 26, 0x000000ff );
    titleTxt.SetAlignment(ALIGN_CENTRE| ALIGN_TOP);
    titleTxt.SetPosition(0,40);

    GuiText msgTxt( font, msg, 22, 0x000000ff);
    msgTxt.SetAlignment( ALIGN_CENTRE | ALIGN_MIDDLE );
    msgTxt.SetPosition(0,-20);
    msgTxt.SetWrap(true, 400);

    GuiImage btn1Img(&btnOutline);
    GuiImage btn1ImgOver(&btnOutlineOver);
	GuiText btn1Txt( font, btn1Label, 22, 0x000000ff);
	btn1Txt.SetPosition( 0, -5 );
    GuiButton btn1(btnOutline.GetWidth(), btnOutline.GetHeight());

    if(btn2Label)
    {
        btn1.SetAlignment(ALIGN_LEFT | ALIGN_BOTTOM);
        btn1.SetPosition(20, -25);
    }
    else
    {
        btn1.SetAlignment(ALIGN_CENTRE | ALIGN_BOTTOM);
        btn1.SetPosition(0, -25);
    }

    btn1.SetLabel(&btn1Txt);
    btn1.SetImage(&btn1Img);
    btn1.SetImageOver(&btn1ImgOver);
	//btn1.SetSoundOver(&btnSoundOver);
    btn1.SetTrigger(&trigA);
    btn1.SetState(STATE_SELECTED);
    btn1.SetEffectGrow();
	btn1.SetSoundOver( &btnSndOver );
	btn1.SetSoundClick( &btnSndClick2 );

    GuiImage btn2Img(&btnOutline);
    GuiImage btn2ImgOver(&btnOutlineOver);
	GuiText btn2Txt( font, btn2Label, 22, 0x000000ff);
	btn2Txt.SetPosition( 0, -5 );
    GuiButton btn2(btnOutline.GetWidth(), btnOutline.GetHeight());
    btn2.SetAlignment(ALIGN_RIGHT| ALIGN_BOTTOM);
    btn2.SetPosition(-20, -25);
    btn2.SetLabel(&btn2Txt);
    btn2.SetImage(&btn2Img);
    btn2.SetImageOver(&btn2ImgOver);
	//btn2.SetSoundOver(&btnSoundOver);
    btn2.SetTrigger(&trigA);
    btn2.SetEffectGrow();
	btn2.SetSoundOver( &btnSndOver );
	btn2.SetSoundClick( &btnSndClick2 );

    promptWindow.Append(&dialogBoxImg);
    promptWindow.Append(&titleTxt);
    promptWindow.Append(&msgTxt);
    promptWindow.Append(&btn1);

    if(btn2Label)
        promptWindow.Append(&btn2);

    promptWindow.SetEffect(EFFECT_SLIDE_TOP | EFFECT_SLIDE_IN, 50);
    HaltGui();
    mainWindow->SetState(STATE_DISABLED);
    mainWindow->Append(&promptWindow);
    mainWindow->ChangeFocus(&promptWindow);
    ResumeGui();

    while(choice == -1)
    {
        usleep(THREAD_SLEEP);

        if(btn1.GetState() == STATE_CLICKED)
            choice = 1;
        else if(btn2.GetState() == STATE_CLICKED)
            choice = 0;
    }
	//let button sounds finish playing
	while( btnSndOver.IsPlaying() || btnSndClick2.IsPlaying() )
		usleep( THREAD_SLEEP );

	//slide window off the screen
    promptWindow.SetEffect(EFFECT_SLIDE_TOP | EFFECT_SLIDE_OUT, 50);
	while( promptWindow.GetEffect() > 0 )
		usleep(THREAD_SLEEP);

    HaltGui();
    mainWindow->Remove(&promptWindow);
    mainWindow->SetState(STATE_DEFAULT);
    ResumeGui();
    return choice;
}

void ProgressWindow( const char *title, const char *msg )
{
	GuiWindow promptWindow( 448,288 );
	promptWindow.SetAlignment( ALIGN_CENTRE | ALIGN_MIDDLE );
	promptWindow.SetPosition( 0, -10 );
//	GuiSound btnSoundOver(button_over_pcm, button_over_pcm_size, SOUND_PCM);
//	GuiSound btnSoundClick(button_click_pcm, button_click_pcm_size, SOUND_PCM);
//    GuiImageData btnOutline( button_png, button_png_size );
//    GuiImageData btnOutlineOver( button_over_png, button_over_png_size );

	GuiImageData dialogBox( dialogue_box_png, dialogue_box_png_size );
	GuiImage dialogBoxImg(&dialogBox);

	GuiImageData progressbarOutline( progressbar_outline_png, progressbar_outline_png_size );
	GuiImage progressbarOutlineImg(&progressbarOutline);
	progressbarOutlineImg.SetAlignment(ALIGN_LEFT | ALIGN_TOP);
	progressbarOutlineImg.SetPosition(23, 170);

	GuiImageData progressbarEmpty( progressbar_empty_png, progressbar_empty_png_size );
	GuiImage progressbarEmptyImg(&progressbarEmpty);
	progressbarEmptyImg.SetAlignment(ALIGN_LEFT | ALIGN_TOP);
	progressbarEmptyImg.SetPosition(23, 170);
	progressbarEmptyImg.SetTile(100);

	GuiImageData progressbar( progressbar_png, progressbar_png_size );
	GuiImage progressbarImg(&progressbar);
	progressbarImg.SetAlignment(ALIGN_LEFT | ALIGN_TOP);
	progressbarImg.SetPosition(23, 170);

	/*GuiImageData throbber(throbber_png);
	GuiImage throbberImg(&throbber);
	throbberImg.SetAlignment(ALIGN_CENTRE, ALIGN_MIDDLE);
	throbberImg.SetPosition(0, 40);*/

	GuiText titleTxt( font, title, 26, 0x000000ff );
	titleTxt.SetAlignment(ALIGN_CENTRE | ALIGN_TOP);
	titleTxt.SetPosition(0,40);

	GuiText msgTxt( font, msg, 22, 0x000000ff );
	msgTxt.SetAlignment(ALIGN_CENTRE | ALIGN_MIDDLE);
	msgTxt.SetPosition(0, -20);
	msgTxt.SetWrap(true, 400);

	promptWindow.Append(&dialogBoxImg);
	promptWindow.Append(&titleTxt);
	promptWindow.Append(&msgTxt);

	promptWindow.Append(&progressbarEmptyImg);
	promptWindow.Append(&progressbarImg);
	promptWindow.Append(&progressbarOutlineImg);


	HaltGui();
	int oldState = mainWindow->GetState();
	mainWindow->SetState(STATE_DISABLED);
	mainWindow->Append(&promptWindow);
	mainWindow->ChangeFocus(&promptWindow);
	ResumeGui();

	float workdone = 0.0;
	float workTotal = 400.0;//some arbatrary abount of something to do

	while( workdone < workTotal )
	{
		//show progress
		progressbarImg.SetTile( 100 * workdone / workTotal );

		//do some imaginary work
		workdone += 0.75f;
		if( workdone > workTotal )
			workdone = workTotal;
		usleep( 5000 );

	}

	HaltGui();
	mainWindow->Remove(&promptWindow);
	mainWindow->SetState(oldState);
	ResumeGui();
}

void InitGameSettings()
{
	gameInfo.player[ 0 ] = HUMAN;
	gameInfo.player[ 1 ] = COMPUTER;
	gameInfo.level = C4_MAX_LEVEL/3;
	strncpy( gameInfo.name[ 0 ], "Red", MAX_PLAYER_NAME_LENGTH );
	strncpy( gameInfo.name[ 1 ], "Black", MAX_PLAYER_NAME_LENGTH );
	gameInfo.turn = 0;
	gameInfo.numPlayers = 1;
	gameInfo.width = 7;
	gameInfo.height = 6;
	gameInfo.numberToConnect = 4;
	gameInfo.first = T_ALTERNATE;
}

void MenuRender()
{
	MenuPrepareToDraw();

	//draw main window
	mainWindow->Draw();

	//draw cursors
	for( int i = 3; i >= 0; i-- )
	{
		if( !userInput[ i ].pad.showCursor )
			continue;

		MenuDrawImage( pointer[ i ]->GetRsxTexOffset(), pointer[ i ]->GetWidth(), pointer[ i ]->GetHeight(),\
					   pointer[ i ]->GetWPitch(), userInput[ i ].pad.cursorX - ( pointer[ i ]->GetWidth() / 2 ),\
					   userInput[ i ].pad.cursorY - ( pointer[ i ]->GetHeight() / 2 ), 0, 0, 0xff, 1.0 );

	}

	tiny3d_Flip();
	FrameTimer++;
}

int MenuMainScreen()
{
	int ret = MENU_EXIT;
	//trigger
	GuiTrigger trigO;
	trigO.SetSimpleTrigger( -1, BTN_CIRCLE_ );
	GuiTrigger trigX;
	trigX.SetSimpleTrigger( -1, BTN_CROSS_ );
	GuiTrigger trigU;
	trigU.SetButtonOnlyTrigger( 0, BTN_UP_ );
	GuiTrigger trigD;
	trigD.SetButtonOnlyTrigger( 0, BTN_DOWN_ );

	//button image
	GuiImageData btnImgData( button_png, button_png_size );
	GuiImageData btnImgOverData( button_over_png, button_over_png_size );

	//button sound
	GuiSound btnSndOver( button_over_wav, button_over_wav_size, SOUND_WAV );
	GuiSound btnSndClick2( button_click2_wav, button_click2_wav_size, SOUND_WAV );
	btnSndClick2.SetVolume( 50 );

	//title text
	char t[ MAX_KEYBOARD_DISPLAY + 1 ];
	strncpy( t, "libps3gui demo", MAX_KEYBOARD_DISPLAY );
	GuiText titleTxt( font, t, 32, GUI_TEXT_COLOR );
	titleTxt.SetPosition( 0, 0 );
	titleTxt.SetAlignment( ALIGN_TOP | ALIGN_CENTER );

	int buttonTop = 110;
	int buttonLeft = 80;
	int buttonX = buttonLeft;
	int buttonY = buttonTop;
	int buttonSpacing = 20;
	int btnTxtY = -5;
	u32 buttonAlignment = ( ALIGN_TOP | ALIGN_LEFT );
	//buttons
	GuiImage buttonImg5( &btnImgData );
	GuiImage buttonOverImg5( &btnImgOverData );
	GuiText c4BtnTxt( font, "Play", buttonImg5.GetHeight() - 22, GUI_TEXT_COLOR );
	c4BtnTxt.SetPosition( 0, btnTxtY );
	GuiButton c4Btn( buttonImg5.GetWidth(), buttonImg5.GetHeight() );
	c4Btn.SetTrigger( &trigX );
	c4Btn.SetImage( &buttonImg5 );
	c4Btn.SetImageOver( &buttonOverImg5 );
	c4Btn.SetPosition( buttonX, buttonY );
	c4Btn.SetAlignment( buttonAlignment );
	c4Btn.SetLabel( &c4BtnTxt );
	c4Btn.SetSoundOver( &btnSndOver );
	c4Btn.SetSoundClick( &btnSndClick2 );
	c4Btn.SetEffect( EFFECT_SLIDE_LEFT | EFFECT_SLIDE_IN, 50 );
	buttonY += buttonImg5.GetHeight() + buttonSpacing;

	GuiImage buttonImg3( &btnImgData );
	GuiImage buttonOverImg3( &btnImgOverData );
	GuiText optionBtnTxt( font, "Options", buttonImg3.GetHeight() - 22, GUI_TEXT_COLOR );
	optionBtnTxt.SetPosition( 0, btnTxtY );
	GuiButton optionBtn( buttonImg3.GetWidth(), buttonImg3.GetHeight() );
	optionBtn.SetTrigger( &trigX );
	optionBtn.SetImage( &buttonImg3 );
	optionBtn.SetImageOver( &buttonOverImg3 );
	optionBtn.SetPosition( buttonX, buttonY );
	optionBtn.SetAlignment( buttonAlignment );
	optionBtn.SetLabel( &optionBtnTxt );
	optionBtn.SetSoundOver( &btnSndOver );
	optionBtn.SetSoundClick( &btnSndClick2 );
	optionBtn.SetEffect( EFFECT_SLIDE_LEFT | EFFECT_SLIDE_IN, 50 );
	buttonY += buttonImg3.GetHeight() + buttonSpacing;

	GuiImage buttonImg2( &btnImgData );
	GuiImage buttonOverImg2( &btnImgOverData );
	GuiText btnTxt( font, "Prompt", buttonImg2.GetHeight() - 22, GUI_TEXT_COLOR );
	btnTxt.SetPosition( 0, btnTxtY );
	GuiButton tBtn( buttonImg2.GetWidth(), buttonImg2.GetHeight() );
	tBtn.SetTrigger( &trigX );
	tBtn.SetImage( &buttonImg2 );
	tBtn.SetImageOver( &buttonOverImg2 );
	tBtn.SetPosition( buttonX, buttonY );
	tBtn.SetAlignment( buttonAlignment );
	tBtn.SetLabel( &btnTxt );
	tBtn.SetSoundOver( &btnSndOver );
	tBtn.SetSoundClick( &btnSndClick2 );
	tBtn.SetEffect( EFFECT_SLIDE_LEFT | EFFECT_SLIDE_IN, 50 );
	buttonY += buttonImg2.GetHeight() + buttonSpacing;

	GuiImage buttonImg4( &btnImgData );
	GuiImage buttonOverImg4( &btnImgOverData );
	GuiText kbBtnTxt( font, "Keyboard", buttonImg4.GetHeight() - 22, GUI_TEXT_COLOR );
	kbBtnTxt.SetPosition( 0, btnTxtY );
	GuiButton kbBtn( buttonImg4.GetWidth(), buttonImg4.GetHeight() );
	kbBtn.SetTrigger( &trigX );
	kbBtn.SetImage( &buttonImg4 );
	kbBtn.SetImageOver( &buttonOverImg4 );
	kbBtn.SetPosition( buttonX, buttonY );
	kbBtn.SetAlignment( buttonAlignment );
	kbBtn.SetLabel( &kbBtnTxt );
	kbBtn.SetSoundOver( &btnSndOver );
	kbBtn.SetSoundClick( &btnSndClick2 );
	kbBtn.SetEffect( EFFECT_SLIDE_LEFT | EFFECT_SLIDE_IN, 50 );
	buttonY += buttonImg4.GetHeight() + buttonSpacing;

	GuiImage buttonImg( &btnImgData );
	GuiImage buttonOverImg( &btnImgOverData );
	GuiText txt( font, "Exit", buttonImg.GetHeight() - 22, GUI_TEXT_COLOR );
	txt.SetPosition( 0, btnTxtY );
	GuiButton exitBtn( buttonImg.GetWidth(), buttonImg.GetHeight() );
	exitBtn.SetTrigger( &trigX );
	exitBtn.SetImage( &buttonImg );
	exitBtn.SetImageOver( &buttonOverImg );
	exitBtn.SetPosition( buttonX, buttonY );
	exitBtn.SetAlignment( buttonAlignment );
	exitBtn.SetLabel( &txt );
	exitBtn.SetSoundOver( &btnSndOver );
	exitBtn.SetSoundClick( &btnSndClick2 );
	exitBtn.SetEffect( EFFECT_SLIDE_LEFT | EFFECT_SLIDE_IN, 50 );
	buttonY += buttonImg.GetHeight() + buttonSpacing;

	//start column 2 for buttons
	buttonY = buttonTop;
	buttonX = buttonLeft + buttonImg.GetWidth() + buttonSpacing;

	GuiImage progBtnImg( &btnImgData );
	GuiImage progBtnOverImg( &btnImgOverData );
	GuiText progBtnTxt( font, "Progress", progBtnImg.GetHeight() - 22, GUI_TEXT_COLOR );
	progBtnTxt.SetPosition( 0, btnTxtY );
	GuiButton progBtn( progBtnImg.GetWidth(), progBtnImg.GetHeight() );
	progBtn.SetTrigger( &trigX );
	progBtn.SetImage( &progBtnImg );
	progBtn.SetImageOver( &progBtnOverImg );
	progBtn.SetPosition( buttonX, buttonY );
	progBtn.SetAlignment( buttonAlignment );
	progBtn.SetLabel( &progBtnTxt );
	progBtn.SetSoundOver( &btnSndOver );
	progBtn.SetSoundClick( &btnSndClick2 );
	progBtn.SetEffect( EFFECT_SLIDE_LEFT | EFFECT_SLIDE_IN, 50 );
	buttonY += buttonImg.GetHeight() + buttonSpacing;

	GuiImage aboutBtnImg( &btnImgData );
	GuiImage aboutBtnOverImg( &btnImgOverData );
	GuiText aboutBtnTxt( font, "About", aboutBtnImg.GetHeight() - 22, GUI_TEXT_COLOR );
	aboutBtnTxt.SetPosition( 0, btnTxtY );
	GuiButton aboutBtn( aboutBtnImg.GetWidth(), aboutBtnImg.GetHeight() );
	aboutBtn.SetTrigger( &trigX );
	aboutBtn.SetImage( &aboutBtnImg );
	aboutBtn.SetImageOver( &aboutBtnOverImg );
	aboutBtn.SetPosition( buttonX, buttonY );
	aboutBtn.SetAlignment( buttonAlignment );
	aboutBtn.SetLabel( &aboutBtnTxt );
	aboutBtn.SetSoundOver( &btnSndOver );
	aboutBtn.SetSoundClick( &btnSndClick2 );
	aboutBtn.SetEffect( EFFECT_SLIDE_LEFT | EFFECT_SLIDE_IN, 50 );
	buttonY += aboutBtnImg.GetHeight() + buttonSpacing;

	GuiWindow w( WINDOW_WIDTH, WINDOW_HEIGHT );
	w.Append( &exitBtn );
	w.Append( &tBtn );
	w.Append( &kbBtn );
	w.Append( &optionBtn );
	w.Append( &c4Btn );
	w.Append( &progBtn );
	w.Append( &aboutBtn );
	w.Append( &titleTxt );

	HaltGui();
	mainWindow->Append( &w );
	ResumeGui();


	while( 1 )
	{
		if( exitBtn.GetState() == STATE_CLICKED )
		{
			exitBtn.ResetState();
			break;
		}
		else if( tBtn.GetState() == STATE_CLICKED )
		{
			tBtn.ResetState();
			WindowPrompt( "This is a prompt", "It tells you important things", "Ok", "Cancel" );
			ErrorPrompt( "This is an error message.  Maybe something bad happened.  Maybe nothing happened.  Who knows?" );
		}
		else if( kbBtn.GetState() == STATE_CLICKED )
		{
			OnScreenKeyboard( t, MAX_KEYBOARD_DISPLAY );
			titleTxt.SetText( t );
			kbBtn.ResetState();
		}
		else if( optionBtn.GetState() == STATE_CLICKED )
		{
			optionBtn.ResetState();
			ret = MENU_SETTINGS;
			break;
		}
		else if( c4Btn.GetState() == STATE_CLICKED )
		{
			c4Btn.ResetState();
			ret = MENU_GAME;
			break;
		}
		else if( progBtn.GetState() == STATE_CLICKED )
		{
			 progBtn.ResetState();
			 ProgressWindow( "ProgressWindow", "Something important might be happening" );
		}
		else if( aboutBtn.GetState() == STATE_CLICKED )
		{
			 aboutBtn.ResetState();
			 ret = MENU_ABOUT;
			 break;

		}

		usleep( THREAD_SLEEP );
	}
	while( btnSndOver.IsPlaying() || btnSndClick2.IsPlaying() )
		usleep( THREAD_SLEEP );

	HaltGui();
	mainWindow->Remove( &w );
	ResumeGui();


	return ret;

}

int MenuSettings()
{
	int ret = MENU_EXIT;
	//trigger
	GuiTrigger trigX;
	trigX.SetSimpleTrigger( -1, BTN_CROSS_ );

	//button image data
	GuiImageData btnImgData( button_png, button_png_size );
	GuiImageData btnImgOverData( button_over_png, button_over_png_size );

	//button sounds
	GuiSound btnSndOver( button_over_wav, button_over_wav_size, SOUND_WAV );
	GuiSound btnSndClick2( button_click2_wav, button_click2_wav_size, SOUND_WAV );
	btnSndClick2.SetVolume( 50 );

	//title text
	GuiText titleTxt( font, "Settings", 32, GUI_TEXT_COLOR );
	titleTxt.SetPosition( 0, 0 );
	titleTxt.SetAlignment( ALIGN_TOP | ALIGN_CENTER );

	//back button
	GuiImage buttonImg( &btnImgData );
	GuiImage buttonOverImg( &btnImgOverData );
	GuiText txt( font, "Done", buttonImg.GetHeight() - 22, GUI_TEXT_COLOR );
	txt.SetPosition( 0, -5 );
	GuiButton exitBtn( buttonImg.GetWidth(), buttonImg.GetHeight() );
	exitBtn.SetTrigger( &trigX );
	exitBtn.SetImage( &buttonImg );
	exitBtn.SetImageOver( &buttonOverImg );
	exitBtn.SetPosition( 100, -80 );
	exitBtn.SetAlignment( ALIGN_BOTTOM | ALIGN_LEFT );
	exitBtn.SetLabel( &txt );
	exitBtn.SetSoundOver( &btnSndOver );
	exitBtn.SetSoundClick( &btnSndClick2 );

	//option list
	OptionList options;
	int numOptions = 0;

	options.SetName( numOptions, "Red Player\'s Name");
	options.SetValue( numOptions, "%s", gameInfo.name[ 0 ] );

	options.SetName( ++numOptions, "Black Player\'s Name");
	options.SetValue( numOptions, "%s", gameInfo.name[ 1 ] );

	options.SetName( ++numOptions, "Player 1");
	if( gameInfo.player[ 0 ] == HUMAN )
		options.SetValue( numOptions, "Human" );
	else
		options.SetValue( numOptions, "CPU" );

	options.SetName( ++numOptions, "Player 2");
	if( gameInfo.player[ 1 ] == HUMAN )
		options.SetValue( numOptions, "Human" );
	else
		options.SetValue( numOptions, "CPU" );

	options.SetName( ++numOptions, "CPU smartitude");
	options.SetValue( numOptions, "%d", gameInfo.level );

	options.SetName( ++numOptions, "Columns");
	options.SetValue( numOptions, "%d", gameInfo.width );

	options.SetName( ++numOptions, "Rows");
	options.SetValue( numOptions, "%d", gameInfo.height );

	options.SetName( ++numOptions, "Length to Win");
	options.SetValue( numOptions, "%d", gameInfo.numberToConnect );

	options.SetName( ++numOptions, "First Move");
	switch( gameInfo.first )
	{
	case T_P1:
		options.SetValue( numOptions, "Red" );
		break;
	case T_P2:
		options.SetValue( numOptions, "Black" );
		break;
	case T_WINNER:
		options.SetValue( numOptions, "Winner" );
		break;
	case T_LOSER:
		options.SetValue( numOptions, "Loser" );
		break;
	default:
		options.SetValue( numOptions, "Alternate" );
		break;
	}

	options.SetName( ++numOptions, "Horizontal Correction");
	options.SetValue( numOptions, "%i", viewportX );

	options.SetName( ++numOptions, "Verticle Correction");
	options.SetValue( numOptions, "%i", viewportY );


	//option browser
	GuiOptionBrowser optionBrowser( 552, 248, &options );
	optionBrowser.SetPosition( 0, 120 );
	optionBrowser.SetAlignment( ALIGN_CENTRE | ALIGN_TOP );
	optionBrowser.SetCol2Position( 260 );
	optionBrowser.SetFocus( 1 );

	GuiWindow w( WINDOW_WIDTH, WINDOW_HEIGHT );
	w.Append( &exitBtn );
	w.Append( &optionBrowser );
	w.Append( &titleTxt );

	HaltGui();
	mainWindow->Append( &w );
	ResumeGui();


	while( 1 )
	{
		if( exitBtn.GetState() == STATE_CLICKED )
		{
			if( gameInfo.level > 10 &&
				( gameInfo.width > 8   || gameInfo.height > 8  ) &&
				( gameInfo.player[ 0 ] || gameInfo.player[ 1 ] ) )
			{
				WindowPrompt( "Notice", \
							  "Setting the CPU level high and using a large board will cause the CPU to take its sweet time deciding on a move.",\
							  "Ok", NULL );
			}
			exitBtn.ResetState();
			ret = MENU_MAIN_SCREEN;
			break;
		}

		int cl = optionBrowser.GetClickedOption();
		switch( cl )
		{
		case 0://edit player names
			{
				HaltGui();
				mainWindow->Remove( &w );
				ResumeGui();
				OnScreenKeyboard( gameInfo.name[ 0 ], MAX_PLAYER_NAME_LENGTH );
				options.SetValue( cl, "%s", gameInfo.name[ 0 ] );
				HaltGui();
				mainWindow->Append( &w );
				ResumeGui();
			}
			break;
		case 1:
			{
				HaltGui();
				mainWindow->Remove( &w );
				ResumeGui();
				OnScreenKeyboard( gameInfo.name[ 1 ], MAX_PLAYER_NAME_LENGTH );
				options.SetValue( cl, "%s", gameInfo.name[ 1 ] );
				HaltGui();
				mainWindow->Append( &w );
				ResumeGui();
			}
			break;
		case 2: //toggle player 1 human/cpu
			{
				gameInfo.player[ 0 ] ^= 1;
				if( gameInfo.player[ 0 ] == HUMAN )
					options.SetValue( cl, "Human" );
				else
					options.SetValue( cl, "CPU" );
			}
			break;
		case 3: //toggle player 2 human/cpu
			{
				gameInfo.player[ 1 ] ^= 1;
				if( gameInfo.player[ 1 ] == HUMAN )
					options.SetValue( cl, "Human" );
				else
					options.SetValue( cl, "CPU" );
			}
			break;
		case 4: //increase difficulty
			{
				if( ++gameInfo.level > C4_MAX_LEVEL )
				{
					gameInfo.level = 1;
				}
				options.SetValue( cl, "%d", gameInfo.level );
			}
			break;
		case 5: //increase columns
			{
				if( ++gameInfo.width > BOARD_MAX_COLS )
				{
					gameInfo.width = BOARD_MIN_COLS;
					int limit = MIN( gameInfo.width, gameInfo.height ) - 1;
					if( gameInfo.numberToConnect >= limit )
					{
						gameInfo.numberToConnect = limit;
						options.SetValue( 7, "%d", gameInfo.numberToConnect );
					}
				}
				options.SetValue( cl, "%d", gameInfo.width );
			}
			break;
		case 6://increase rows
			{
				if( ++gameInfo.height > BOARD_MAX_ROWS )
				{
					gameInfo.height = BOARD_MIN_ROWS;
					int limit = MIN( gameInfo.width, gameInfo.height ) - 1;
					if( gameInfo.numberToConnect >= limit )
					{
						gameInfo.numberToConnect = limit;
						options.SetValue( 7, "%d", gameInfo.numberToConnect );
					}
				}
				options.SetValue( cl, "%d", gameInfo.height );
			}
			break;
		case 7://length to win
			{
				if( ++gameInfo.numberToConnect > gameInfo.width - 1
					&&  gameInfo.numberToConnect > gameInfo.height - 1 )
					gameInfo.numberToConnect = 3;
				options.SetValue( cl, "%d", gameInfo.numberToConnect );
			}
			break;
		case 8://who goes first
			{
				if( ++gameInfo.first >= T_MAX )
					gameInfo.first = T_ALTERNATE;
				switch( gameInfo.first )
				{
				case T_P1:
					options.SetValue( cl, "Red" );
					break;
				case T_P2:
					options.SetValue( cl, "Black" );
					break;
				case T_WINNER:
					options.SetValue( cl, "Winner" );
					break;
				case T_LOSER:
					options.SetValue( cl, "Loser" );
					break;
				default:
					options.SetValue( cl, "Alternate" );
					break;
				}
			}
			break;
		case 9://h correction
			{
				viewportX += 10;
				if( viewportX > VIEWPORT_MAX )
					viewportX = VIEWPORT_MIN;
				AdjustViewport( viewportX, viewportY );
				options.SetValue( cl, "%i", viewportX );
			}
			break;
		case 10://verticle correction
			{
				viewportY += 10;
				if( viewportY > VIEWPORT_MAX )
					viewportY = VIEWPORT_MIN;
				AdjustViewport( viewportX, viewportY );
				options.SetValue( cl, "%i", viewportY );

			}
			break;
		default:
			break;
		}
		if( cl != -1 )
			optionBrowser.TriggerUpdate();

		usleep( THREAD_SLEEP );
	}
	//wait for button sound effects to play before destroying them
	while( btnSndOver.IsPlaying() || btnSndClick2.IsPlaying() )
		usleep( THREAD_SLEEP );

	HaltGui();
	mainWindow->Remove( &w );
	ResumeGui();

	return ret;

}

int MenuConnect4()
{
	int ret = 0;
	char buf[ 0x100 ];

	//make sure settings are in range
	if( gameInfo.width > BOARD_MAX_COLS || gameInfo.width < BOARD_MIN_COLS
		|| gameInfo.height > BOARD_MAX_ROWS || gameInfo.height < BOARD_MIN_ROWS
		|| gameInfo.numberToConnect >= gameInfo.height || gameInfo.numberToConnect >= gameInfo.width )
	{
		ErrorPrompt( "The game parameters are out of range.  Please try again." );
		return MENU_MAIN_SCREEN;
	}

	//window
	GuiWindow w( WINDOW_WIDTH, WINDOW_HEIGHT );

	//button sounds
	GuiSound btnSndOver( button_over_wav, button_over_wav_size, SOUND_WAV );
	GuiSound btnSndClick2( button_click2_wav, button_click2_wav_size, SOUND_WAV );
	GuiSound btnSndClick( button_click_wav, button_click_wav_size, SOUND_WAV );
	btnSndClick2.SetVolume( 50 );

	//text
	GuiText header( font, NULL, 32, GUI_TEXT_COLOR );
	header.SetAlignment( ALIGN_TOP | ALIGN_CENTER );
	header.SetPosition( 0, 10 );
	snprintf( buf, sizeof( buf ), "Waiting for %s to move...", ( gameInfo.turn ? "black" : "red" ) );
	header.SetText( buf );

	snprintf( buf, sizeof( buf ), "%d x %d  -  connect %d to win", gameInfo.width, gameInfo.height, gameInfo.numberToConnect  );
	GuiText info( font, buf, 12, GUI_TEXT_COLOR );
	info.SetAlignment( ALIGN_TOP | ALIGN_RIGHT );
	info.SetPosition( -10, 50 );

	snprintf( buf, sizeof( buf ), "red: %u   black: %u   ties: %u", gameInfo.p1Score, gameInfo.p2Score, gameInfo.ties  );
	GuiText score( font, buf, 12, GUI_TEXT_COLOR );
	score.SetAlignment( ALIGN_TOP | ALIGN_RIGHT );
	score.SetPosition( -10, 68 );

	//triggers
	GuiTrigger trig1x;
	trig1x.SetSimpleTrigger( 0, BTN_CROSS_ );
	GuiTrigger trig2x;
	trig2x.SetSimpleTrigger( 1, BTN_CROSS_ );
	GuiTrigger trigSt;
	trigSt.SetButtonOnlyTrigger( -1, BTN_START_ );

	//image data
	GuiImageData boardImgData( boardSprite_png, boardSprite_png_size );
	GuiImageData redImgData( redPiece_png, redPiece_png_size );
	GuiImageData blackImgData( blackPiece_png, blackPiece_png_size );

	//setup gui components
	int boardHeight = 0;
	int boardWidth = 0;
	int boardLeft = 0;
	int boardTop = 0;
	int cellSize = 0;
	int rowAdj = 0;

	//frame
	int horLen;
	int framethickness = 15;

	int pieceCnt = ( gameInfo.width * gameInfo.height / 2 ) + 1;
	int widthFrowCols = ( WINDOW_WIDTH ) / gameInfo.width;
	int widthFromRows = ( WINDOW_HEIGHT ) / ( gameInfo.height + 1 );
	cellSize = MIN( widthFrowCols, widthFromRows );
	cellSize = MIN( cellSize, boardImgData.GetWidth() );
	float scale = (float)cellSize/(float)boardImgData.GetWidth();
	rowAdj = ( boardImgData.GetWidth() - cellSize ) / 2;



	boardHeight = cellSize * ( gameInfo.height + 1 );
	boardWidth = cellSize * gameInfo.width;

	boardTop = ( ( WINDOW_HEIGHT ) - boardHeight ) / 2;
	boardLeft = ( ( WINDOW_WIDTH ) - boardWidth ) / 2;

	//raise the board up just a little bit
	boardTop -= framethickness;

	//create sprites
	GuiImage *boardImg[ gameInfo.width * gameInfo.height ];
	GuiImage *redImg[ pieceCnt ];
	GuiImage *blackImg[ pieceCnt ];


	//create checkers
	for( int i = 0; i < pieceCnt; i++ )
	{
		redImg[ i ] = new GuiImage( &redImgData );
		redImg[ i ]->SetPosition( -cellSize, boardTop );
		redImg[ i ]->SetVisible( false );
		redImg[ i ]->SetScale( scale );
		w.Append( redImg[ i ] );

		blackImg[ i ] = new GuiImage( &blackImgData );
		blackImg[ i ]->SetPosition( -cellSize, boardTop );
		blackImg[ i ]->SetVisible( false );
		blackImg[ i ]->SetScale( scale );
		w.Append( blackImg[ i ] );
	}

	//create board
	int idx = 0;
	for( int r = 1; r <= gameInfo.height; r++ )
	{
		//printf("putting row %i at %i  ( %i + ( %i * %i ) )\n", r , boardTop + ( cellSize * r ), boardTop, cellSize, r );
		for( int c = 0; c < gameInfo.width; c++ )
		{
			boardImg[ idx ] = new GuiImage( &boardImgData );
			boardImg[ idx ]->SetScale( scale );
			boardImg[ idx ]->SetPosition( boardLeft + ( cellSize * c ), ( boardTop + ( cellSize * r ) ) - rowAdj );
			w.Append( boardImg[ idx ] );

			idx++;
		}
	}

	//create frame for board
	horLen = boardWidth + 40;

#define BOARD_H_COLOR 0x826526ff
#define BOARD_V_COLOR 0xbe9439ff

	GuiImage top( horLen, framethickness, BOARD_H_COLOR );
	top.SetAlignment( ALIGN_CENTER | ALIGN_TOP );
	top.SetPosition( 0, ( boardTop + cellSize + 1 ) - framethickness );


	GuiImage bottom( horLen, framethickness, BOARD_H_COLOR );
	bottom.SetAlignment( ALIGN_CENTER | ALIGN_TOP );
	bottom.SetPosition( 0, boardTop + boardHeight );

	GuiImage left( framethickness, ( boardHeight - cellSize ), BOARD_V_COLOR );
	left.SetAlignment( ALIGN_LEFT | ALIGN_TOP );
	left.SetPosition( boardLeft - framethickness, ( boardTop + cellSize ) );

	GuiImage right( framethickness, ( boardHeight - cellSize ), BOARD_V_COLOR );
	right.SetAlignment( ALIGN_LEFT | ALIGN_TOP );
	right.SetPosition( boardLeft + boardWidth, ( boardTop + cellSize ) );


	/*printf("putting board top at %i ( ( %i + %i + 1 ) - %i ) \n", ( boardTop + cellSize + 1 ) - framethickness\
		   , boardTop, cellSize, framethickness );
	printf("putting board bottom at %i\n", boardTop + boardHeight );*/
	//fflush( stdout );

	w.Append( &left );
	w.Append( &right );
	w.Append( &top );
	w.Append( &bottom );

	//create buttons to detect a click
	GuiButton *redButton[ gameInfo.width ];
	GuiButton *blackButton[ gameInfo.width ];
	for( int i = 0; i < gameInfo.width; i++ )
	{
		redButton[ i ] = new GuiButton( cellSize, boardHeight );
		redButton[ i ]->SetPosition( boardLeft + ( cellSize * i ), boardTop );
		redButton[ i ]->SetTrigger( &trig1x );

		blackButton[ i ] = new GuiButton( cellSize, boardHeight );
		blackButton[ i ]->SetPosition( boardLeft + ( cellSize * i ), boardTop );
		blackButton[ i ]->SetTrigger( &trig2x );

		w.Append( redButton[ i ] );
		w.Append( blackButton[ i ] );
	}

	//other buttons
	//invisible exit button
	GuiButton startBtn( 0, 0 );
	startBtn.SetTrigger( &trigSt );
	w.Append( &startBtn );

	//restart game button
	//button image data
	GuiImageData btnImgData( button_png, button_png_size );
	GuiImageData btnImgOverData( button_over_png, button_over_png_size );

	//between-game buttons
	int buttonY = 0;
	int buttonSpacing = 10;
	GuiImage restartBtnImg( &btnImgData );
	GuiImage restartBtnOverImg( &btnImgOverData );
	restartBtnImg.SetAlpha( 180 );
	restartBtnOverImg.SetAlpha( 180 );
	GuiText restartBtnTxt( font, "Restart", restartBtnImg.GetHeight() - 22, GUI_TEXT_COLOR );
	GuiButton restartBtn( restartBtnImg.GetWidth(), restartBtnImg.GetHeight() );
	restartBtnTxt.SetPosition( 0, -5 );
	restartBtn.SetTrigger( &trig1x );
	restartBtn.SetTrigger( &trig2x );
	restartBtn.SetImage( &restartBtnImg );
	restartBtn.SetImageOver( &restartBtnOverImg );
	restartBtn.SetPosition( 0, buttonY );
	restartBtn.SetAlignment( ALIGN_CENTER | ALIGN_MIDDLE );
	restartBtn.SetLabel( &restartBtnTxt );
	restartBtn.SetVisible( false );
	restartBtn.SetSoundOver( &btnSndOver );
	restartBtn.SetSoundClick( &btnSndClick2 );
	buttonY += restartBtnImg.GetHeight() + buttonSpacing;

	GuiImage backBtnImg( &btnImgData );
	GuiImage backBtnOverImg( &btnImgOverData );
	backBtnImg.SetAlpha( 180 );
	backBtnOverImg.SetAlpha( 180 );
	GuiText backBtnTxt( font, "Main Menu", backBtnImg.GetHeight() - 22, GUI_TEXT_COLOR );
	backBtnTxt.SetPosition( 0, -5 );
	GuiButton backBtn( backBtnImg.GetWidth(), backBtnImg.GetHeight() );
	backBtn.SetTrigger( &trig1x );
	backBtn.SetTrigger( &trig2x );
	backBtn.SetImage( &backBtnImg );
	backBtn.SetImageOver( &backBtnOverImg );
	backBtn.SetPosition( 0, buttonY );
	backBtn.SetAlignment( ALIGN_CENTER | ALIGN_MIDDLE );
	backBtn.SetLabel( &backBtnTxt );
	backBtn.SetVisible( false );
	backBtn.SetSoundOver( &btnSndOver );
	backBtn.SetSoundClick( &btnSndClick2 );
	buttonY += backBtnImg.GetHeight() + buttonSpacing;

	w.Append( &header );
	w.Append( &info );
	w.Append( &score );
	w.Append( &restartBtn );
	w.Append( &backBtn );


	HaltGui();
	mainWindow->Append( &w );
	ResumeGui();

	//start a game

	if( gameInfo.first == T_P1 )
		gameInfo.turn = 0;
	else if( gameInfo.first == T_P2 )
		gameInfo.turn = 1;

	int mode = gameInfo.player[ gameInfo.turn ] == HUMAN ? M_WAIT_PLAYER : M_WAIT_CPU;
	int cycle = 0;
	c4_new_game( gameInfo.width, gameInfo.height, gameInfo.numberToConnect );



	//use when animating pieces
	GuiImage *checker = NULL;
	int destX = 0;
	int destY = 0;

	//play the game
	while( !ret )
	{
		usleep( THREAD_SLEEP );

		//exit
		if( startBtn.GetState() == STATE_CLICKED )
		{
			startBtn.ResetState();
			ret = MENU_MAIN_SCREEN;
			break;
		}

		switch( mode )
		{
		case M_WAIT_PLAYER://we are waiting for a human to make a move
			{
				//check the buttons to see if they have clicked
				for( int i = 0; i < gameInfo.width; i++ )
				{
					if( redButton[ i ]->GetState() == STATE_CLICKED )
					{
						redButton[ i ]->ResetState();

						//player 1 ( red ) has clicked a button and it is their turn
						if( !gameInfo.turn )
						{
							int row;
							bool ok = c4_make_move( gameInfo.turn, i, &row );
							if( !ok )
							{
								ErrorPrompt( "You can\'t move there" );
								break;
								//exit( 0 );
							}
							//set image visible and signal animate
							checker = redImg[ cycle ];
							checker->SetVisible( true );
							destX = boardLeft + ( i * cellSize );
							destY = boardTop + ( ( ( gameInfo.height - row ) * cellSize ) - rowAdj );
							mode = M_ANIMATE;
						}
						break;
					}
					if( blackButton[ i ]->GetState() == STATE_CLICKED )
					{
						blackButton[ i ]->ResetState();
						//player 2 ( black ) has clicked a button and it is their turn
						if( gameInfo.turn )
						{
							int row;
							bool ok = c4_make_move( gameInfo.turn, i, &row );
							if( !ok )
							{
								ErrorPrompt( "You can\'t move there" );
								break;
							}
							//set image visible and signal animate
							checker = blackImg[ cycle ];
							checker->SetVisible( true );
							destX = boardLeft + ( i * cellSize );
							destY = boardTop + ( ( ( gameInfo.height - row ) * cellSize ) - rowAdj );
							mode = M_ANIMATE;
						}
						break;
					}
				}
			}
			break;
		case M_WAIT_CPU://it is the CPU's turn to move
			{
				int col;
				int row;
				bool ok = c4_auto_move( gameInfo.turn, gameInfo.level, &col, &row );
				if( !ok )
				{
					ErrorPrompt( "Something went wrong trying to make that move and I dont have any error handeling in place." );
					exit( 0 );
				}

				//set image visible and signal animate
				if( !gameInfo.turn )
					checker = redImg[ cycle ];
				else
					checker = blackImg[ cycle ];

				checker->SetVisible( true );
				destX = boardLeft + ( col * cellSize );
				destY = boardTop + ( ( ( gameInfo.height - row ) * cellSize ) - rowAdj );
				mode = M_ANIMATE;

				//clear any user button presses
				for( int i = 0; i < gameInfo.width; i++ )
				{
					redButton[ i ]->ResetState();
					blackButton[ i ]->ResetState();
				}
			}
			break;
		case M_ANIMATE://slide a piece into place
			{
				//slow animation down a bit
				usleep( 200 );

				int left = checker->GetLeft();
				int top = checker->GetTop();

				//slide right
				if( left < destX )
				{
					checker->SetPosition( left + 1, top );
					break;
				}

				//drop down
				if( top < destY )
				{
					checker->SetPosition( left, top + 1 );
					break;
				}
				//if it makes it to this point, the checker is in place.  play the sound
				btnSndClick.Play();

				//signal next mode
				mode = M_CHECK_GAME_END;
			}
			break;
		case M_CHECK_GAME_END:
			{
				int winner = -1;
				if( c4_is_winner( 0 ) )
				{
					winner = 0;
					gameInfo.p1Score++;
				}
				else if( c4_is_winner( 1 ) )
				{
					winner = 1;
					gameInfo.p2Score++;
				}
				else if ( c4_is_tie() )
				{
					winner = 2;
					gameInfo.ties++;
				}
				if( winner == -1 )//game is still going on
				{
					//black just went, so increase this to use the next checkers for each player
					if( gameInfo.turn )
						cycle++;

					//switch to the other player
					gameInfo.turn ^= 1;

					//set the mode flag
					mode = gameInfo.player[ gameInfo.turn ] == HUMAN ? M_WAIT_PLAYER : M_WAIT_CPU;

					snprintf( buf, sizeof( buf ), "Waiting for %s to move...", ( gameInfo.turn ? "black" : "red" ) );
					header.SetText( buf );
					break;
				}


				//update score and text
				snprintf( buf, sizeof( buf ), "red: %u   black: %u   ties: %u", gameInfo.p1Score, gameInfo.p2Score, gameInfo.ties  );
				HaltGui();
				header.SetText( "Game Over" );
				score.SetText( buf );
				ResumeGui();
				//hide the board during prompts since 300 sprites on the screen at once lags the gui thread a little bit
				//HaltGui();
				//w.SetVisible( false );
				//ResumeGui();
				if( gameInfo.player[ gameInfo.turn ] == COMPUTER
					&& gameInfo.player[ !gameInfo.turn ] == HUMAN
					&& winner != 2 )
				{
					WindowPrompt( "Loser!!", "You lost.  Maybe you should try again on a lower difficulty.", "Ok", NULL );
				}
				else if( winner != 2 )
				{
					snprintf( buf, sizeof( buf ), "%s is the winner.", gameInfo.name[ winner ] );
					WindowPrompt( "Winner!!", buf, "Ok",NULL );
				}
				else
				{
					WindowPrompt( "Losers!!", "It's a tie.  Both players suck.", "Ok", NULL );
				}

				//switch the player that starts the game
				switch( gameInfo.first )
				{
				case T_P1:
					gameInfo.turn = 0;
					break;
				case T_P2:
					gameInfo.turn = 1;
					break;
				case T_WINNER:
					break;
				case T_LOSER:
				default:
					gameInfo.turn ^= 1;
					break;
				}

				HaltGui();
				restartBtn.SetVisible( true );
				backBtn.SetVisible( true );
				ResumeGui();
				mode = M_WAIT_NEW_GAME;
			}
			break;
		case M_RESET_BOARD://something told us to reset the board
			{
				//end current game
				c4_end_game();

				//reset checkers
				cycle = 0;
				for( int i = 0; i < pieceCnt; i++ )
				{
					redImg[ i ]->SetPosition( -cellSize, boardTop );
					redImg[ i ]->SetVisible( false );

					blackImg[ i ]->SetPosition( -cellSize, boardTop );
					blackImg[ i ]->SetVisible( false );
				}

				//unclick all buttons
				for( int i = 0; i < gameInfo.width; i++ )
				{
					redButton[ i ]->ResetState();
					blackButton[ i ]->ResetState();
				}

				//start a new game
				c4_new_game( gameInfo.width, gameInfo.height, gameInfo.numberToConnect );

				//set the mode flag
				mode = gameInfo.player[ gameInfo.turn ] == HUMAN ? M_WAIT_PLAYER : M_WAIT_CPU;

				snprintf( buf, sizeof( buf ), "Waiting for %s to move...", ( gameInfo.turn ? "black" : "red" ) );

				HaltGui();
				header.SetText( buf );
				ResumeGui();
			}
			break;
		case M_WAIT_NEW_GAME:
			{
				//restart
				if( restartBtn.GetState() == STATE_CLICKED )
				{
					restartBtn.ResetState();

					HaltGui();
					restartBtn.SetVisible( false );
					backBtn.SetVisible( false );
					ResumeGui();

					//set the mode flag
					mode = M_RESET_BOARD;
				}
				else if( backBtn.GetState() == STATE_CLICKED )
				{
					backBtn.ResetState();
					ret = MENU_MAIN_SCREEN;
				}
			}
			break;

		default:
			break;

		}
	}

	//end current game
	c4_end_game();

	//wait for button sound effects to play before destroying them
	while( btnSndOver.IsPlaying() || btnSndClick2.IsPlaying() || btnSndClick.IsPlaying() )
		usleep( THREAD_SLEEP );

	//clean up
	HaltGui();
	mainWindow->Remove( &w );

	for( int i = 0; i < gameInfo.width; i++ )
	{
		delete redButton[ i ];
		delete blackButton[ i ];
	}

	idx = gameInfo.height * gameInfo.width ;
	for( int c = 0; c < idx; c++ )
	{
		delete boardImg[ c ];
	}

	for( int i = 0; i < pieceCnt; i++ )
	{
		delete redImg[ i ];
		delete blackImg[ i ];
	}
	ResumeGui();

	//return
	return ret;
}

int MenuAbout()
{
	GuiWindow promptWindow( WINDOW_WIDTH, WINDOW_HEIGHT );
	promptWindow.SetAlignment( ALIGN_CENTRE | ALIGN_MIDDLE );


	GuiTrigger trigA;
	trigA.SetButtonOnlyInFocusTrigger( -1, BTN_CROSS_ );

	GuiImageData dialogBox( aboutWindow_png, aboutWindow_png_size );
	GuiImage dialogBoxImg( &dialogBox );
	dialogBoxImg.SetPosition( -5, -5 );//the image is not centered due to the shadow i included in it
	dialogBoxImg.SetAlignment( ALIGN_CENTER | ALIGN_MIDDLE );

	GuiText titleTxt( font, "libps3gui demo", 30, 0x000000ff  );
	titleTxt.SetAlignment(ALIGN_CENTRE| ALIGN_TOP);
	titleTxt.SetPosition( 0,40 );


	char msg[ 2048 ];
	snprintf( msg, sizeof( msg ), \
			  "This is a quick demo for libps3gui.  The library is based off libwiigui ((c) Tantric 2009).  "\
			  "This ps3 port is done by giantpune, including code from hermes and dimok.  "\
			  "It uses tiny3d for drawing.  Controller input and drawing is done on 1 thread.  "\
			  "Sound is played via hermes\' spu module and the background music is being converted from ogg "\
			  "to pcm on a thread.  The API is very similar to libwiigui.  "\
			  "The connect 4 game engine is written by Keith Pomakis.  "\
			  "This code is licensed under the GPLv2 license.");
	GuiText msgTxt( font, msg, 24, 0x000000ff );
	msgTxt.SetAlignment( ALIGN_CENTER | ALIGN_TOP );
	msgTxt.SetPosition( 0, 120 );
	msgTxt.SetWrap( true, dialogBoxImg.GetWidth() - 50 );


	GuiButton btn1( 0, 0 );
	btn1.SetTrigger( &trigA );


	promptWindow.Append( &dialogBoxImg );
	promptWindow.Append( &titleTxt );
	promptWindow.Append( &msgTxt );
	promptWindow.Append( &btn1 );

	promptWindow.SetEffect(EFFECT_SLIDE_TOP | EFFECT_SLIDE_IN, 50);
	HaltGui();
	mainWindow->SetState(STATE_DISABLED);
	mainWindow->Append(&promptWindow);
	mainWindow->ChangeFocus(&promptWindow);
	ResumeGui();

	while( 1 )
	{
		usleep( THREAD_SLEEP );

		if( btn1.GetState() == STATE_CLICKED )
			break;
	}

	//slide window off the screen
	promptWindow.SetEffect(EFFECT_SLIDE_TOP | EFFECT_SLIDE_OUT, 50);
	while( promptWindow.GetEffect() > 0 )
		usleep(THREAD_SLEEP);

	HaltGui();
	mainWindow->Remove(&promptWindow);
	mainWindow->SetState(STATE_DEFAULT);
	ResumeGui();
	return MENU_MAIN_SCREEN;
}

int MainMenu( int menu )
{
	//create cursors
	pointer[ 0 ] = new GuiImageData( player1_point_png, player1_point_png_size );
	pointer[ 1 ] = new GuiImageData( player2_point_png, player2_point_png_size );
	pointer[ 2 ] = new GuiImageData( player3_point_png, player3_point_png_size );
	pointer[ 3 ] = new GuiImageData( player4_point_png, player4_point_png_size );

	//create main window, font, background sound, and background
	mainWindow = new GuiWindow( WINDOW_WIDTH, WINDOW_HEIGHT );
	bgImgData = new GuiImageData( background_png, background_png_size );
	bgImage = new GuiImage( bgImgData );
	bgImage->SetAlignment( ALIGN_CENTER | ALIGN_MIDDLE );
	font = new GuiFont( (u8*)font_ttf, font_ttf_size );

	bgMusic = new GuiSound( (const u8*)bg_music_ogg, bg_music_ogg_size, SOUND_OGG );
	bgMusic->SetLoop( true );
	bgMusic->SetVolume( 10 );
	bgMusic->Play();

	mainWindow->Append( bgImage );

	//printf("all global elements created.  starting up gui thread\n");
	ResumeGui();

	while( menu != MENU_EXIT )
	{
		switch( menu )
		{
		case MENU_SETTINGS:
			menu = MenuSettings();
			break;
		case MENU_MAIN_SCREEN:
			menu = MenuMainScreen();
			break;
		case MENU_GAME:
			menu = MenuConnect4();
			break;
		case MENU_ABOUT:
			menu = MenuAbout();
			break;
		default:
			break;
		}
	}
	return menu;
}

void MenuPrepareToDraw()
{
	// clear the screen, buffer Z and initializes environment to 2D
	tiny3d_Clear(0xff000000, TINY3D_CLEAR_ALL);

	// Enable alpha Test
	tiny3d_AlphaTest(1, 0x10, TINY3D_ALPHA_FUNC_GEQUAL);

	// Enable alpha blending.
	tiny3d_BlendFunc(1, (blend_src_func)(TINY3D_BLEND_FUNC_SRC_RGB_SRC_ALPHA | TINY3D_BLEND_FUNC_SRC_ALPHA_SRC_ALPHA),
		(blend_dst_func)(NV30_3D_BLEND_FUNC_DST_RGB_ONE_MINUS_SRC_ALPHA | NV30_3D_BLEND_FUNC_DST_ALPHA_ZERO),
		(blend_func)(TINY3D_BLEND_RGB_FUNC_ADD | TINY3D_BLEND_ALPHA_FUNC_ADD));
}

void MenuFillBackground( u32 rgba )
{
	tiny3d_SetPolygon( TINY3D_QUADS );

	tiny3d_VertexPos( 0, 0, 65535 );
	tiny3d_VertexColor( rgba );

	tiny3d_VertexPos( WINDOW_WIDTH, 0, 65535 );
	tiny3d_VertexPos( WINDOW_WIDTH, WINDOW_HEIGHT, 65535 );
	tiny3d_VertexPos( 0  , WINDOW_HEIGHT, 65535 );
	tiny3d_End();
}

//yet untested
void MenuDrawRectangleRot( u32 rgba, float width, float height, float x, float y, float z, u8 filled, float angleZ, float scale )
{
	//printf("MenuDrawRectangle( %08x, %.2f, %.2f, %.2f, %.2f, %.2f, %08x, %08x )\n", rgba, width, height, x, y, z, alignment, filled);
	//vars

	float dx = ( width / 2 ) * scale;
	float dy = ( height / 2 ) * scale;

	MATRIX matrix;

	float angle = -( PI * angleZ )/180.0f;

	// rotate and translate the sprite
	matrix = MatrixRotationZ( angle );
	matrix = MatrixMultiply( matrix, MatrixTranslation(x + dx, y + dy, 0.0f ) );

	VECTOR tr;
	VECTOR tl;
	VECTOR bl;
	VECTOR br;

	tl.x = -dx;
	tl.y = -dy;
	tl.z = z;

	//assign all other corners based on the top left one
	tr.x = dx;
	tr.y = -dy;
	tr.z = z;

	br.x = dx;
	br.y = dy;
	br.z = z;

	bl.x = -dx;
	bl.y = dy;
	bl.z = z;

	//start drawing
	if( filled )
		tiny3d_SetPolygon( TINY3D_QUADS );
	else
		tiny3d_SetPolygon( TINY3D_LINE_LOOP );

	tiny3d_VertexPosVector( tl );
	tiny3d_VertexColor( rgba );
	tiny3d_VertexPosVector( tr );
	tiny3d_VertexPosVector( br );
	tiny3d_VertexPosVector( bl );

	tiny3d_End();
	tiny3d_SetMatrixModelView(NULL); // set matrix identity
}

void MenuDrawRectangle( u32 rgba, float width, float height, float x, float y, float z, u8 filled, float angleZ, float scale )
{
	//printf("MenuDrawRectangle( %08x, %.2f, %.2f, %.2f, %.2f, %.2f, %u )\n", rgba, width, height, x, y, z, filled);
	if( angleZ )
	{
		MenuDrawRectangleRot( rgba,  width, height, x, y, z, filled, angleZ, scale );
		return;
	}
	//vars
	VECTOR tr;
	VECTOR tl;
	VECTOR bl;
	VECTOR br;

	float dx = x + width;
	float dy = y + height;

	if( scale != 1.0f )
	{
		float difX = ( ( width * scale ) - width ) / 2.0f;
		dx += difX;
		x -= difX;

		float difY = ( ( height * scale ) - height ) / 2.0f;
		dy += difY;
		y -= difY;
	}

	tl.x = x;
	tl.y = y;
	tl.z = z;

	//assign all other corners based on the top left one
	tr.x = dx;
	tr.y = y;
	tr.z = z;

	br.x = dx;
	br.y = dy;
	br.z = z;

	bl.x = x;
	bl.y = dy;
	bl.z = z;

	//start drawing
	if( filled )
		tiny3d_SetPolygon( TINY3D_QUADS );
	else
		tiny3d_SetPolygon( TINY3D_LINE_LOOP );

	tiny3d_VertexPosVector( tl );
	tiny3d_VertexColor( rgba );
	tiny3d_VertexPosVector( tr );
	tiny3d_VertexPosVector( br );
	tiny3d_VertexPosVector( bl );

	tiny3d_End();
}

void MenuDrawImageRot( u32 rsxOffset, float width, float height, u32 wpitch, float x, float y, float z, float angleZ, u8 alpha, float scale )
{
    //printf("MenuDrawImageRot( %08x, %.2f, %.2f, %08x, %.2f, %.2f, %.2f, %.2f )\n",rsxOffset, width, height, wpitch, x, y, z, angleZ );
    float dx = ( width / 2 ) * scale;
    float dy = ( height / 2 ) * scale;

    tiny3d_SetTexture( 0, rsxOffset, width, height, wpitch, TINY3D_TEX_FORMAT_A8R8G8B8, TEXTURE_LINEAR );

    MATRIX matrix;

    float angle = -( PI * angleZ )/180.0f;

    // rotate and translate the sprite
    matrix = MatrixRotationZ( angle );
    matrix = MatrixMultiply( matrix, MatrixTranslation(x + dx, y + dy, 0.0f ) );


    // fix ModelView Matrix
    tiny3d_SetMatrixModelView( &matrix );

    tiny3d_SetPolygon( TINY3D_QUADS );

    tiny3d_VertexPos( -dx, -dy, z );
	tiny3d_VertexColor( 0xffffff00 | alpha );
	tiny3d_VertexTexture( 0.0f ,0.0f );

	tiny3d_VertexPos(dx , -dy, z );
	tiny3d_VertexTexture( 1, 0.0f );

	tiny3d_VertexPos(dx , dy , z );
	tiny3d_VertexTexture( 1, 1 );

	tiny3d_VertexPos(-dx, dy , z );
	tiny3d_VertexTexture( 0.0f ,1 );
	/*

	tiny3d_VertexTexture(0.0f , 0.0f);

	tiny3d_VertexPos(dx , -dy, z );
	tiny3d_VertexTexture(0.99f, 0.0f);

	tiny3d_VertexPos(dx , dy , z );
	tiny3d_VertexTexture(0.99f, 0.99f);

	tiny3d_VertexPos(-dx, dy , z );
	tiny3d_VertexTexture(0.0f , 0.99f);
	*/

    tiny3d_End();

    tiny3d_SetMatrixModelView(NULL); // set matrix identity
}

void MenuDrawImage( u32 rsxOffset, float width, float height, u32 wpitch, float x, float y, float z, float angleZ, u8 alpha, float scale )
{
    if( angleZ )
    {
        MenuDrawImageRot( rsxOffset, width, height, wpitch, x, y, z, angleZ, alpha, scale );
        return;
    }
    //printf("wtf\n");
    //printf("MenuDrawImage( %08x, %.2f, %.2f, %08x, %.2f, %.2f, %.2f, %u, %.2f )\n",rsxOffset, width, height, wpitch, x, y, z, alpha, scale );

    tiny3d_SetTexture( 0, rsxOffset, width, height, wpitch, TINY3D_TEX_FORMAT_A8R8G8B8, TEXTURE_LINEAR );

    tiny3d_SetPolygon( TINY3D_QUADS );

    float dx = x + width;
    float dy = y + height;

    if( scale != 1.0f )
    {
        float difX = ( ( width * scale ) - width ) / 2.0f;
        dx += difX;
        x -= difX;

        float difY = ( ( height * scale ) - height ) / 2.0f;
        dy += difY;
        y -= difY;
    }

	tiny3d_VertexPos( x, y, z );
	tiny3d_VertexColor( 0xffffff00 | alpha );
	tiny3d_VertexTexture( 0.0f, 0.0f );

	tiny3d_VertexPos( dx, y, z );
	tiny3d_VertexTexture(1, 0.0f);

	tiny3d_VertexPos( dx, dy, z );
	tiny3d_VertexTexture(1, 1);

	tiny3d_VertexPos( x, dy, z );
	tiny3d_VertexTexture(0.0f, 1);

    tiny3d_End();
}

void AdjustViewport( float x, float y )
{
	double sx = (double) Video_Resolution.width;
	double sy = (double) Video_Resolution.height;
	double px = (double) (1000 + x)/1000.0;
	double py = (double) (1000 + y)/1000.0;

	tiny3d_UserViewport( 1,
						(float) ((sx - sx * px) / 2.0), // 2D position
						(float) ((sy - sy * py) / 2.0),
						(float) ((sx * px) / 848.0),    // 2D scale
						(float) ((sy * py) / 512.0),
						(((float) Video_Resolution.width) / 1920.0f) * (float) (1000 + x)/1000.0f,  // 3D scale
						(((float) Video_Resolution.height) / 1080.0f) * (float) (1000 + y)/1000.0f);



}

