/****************************************************************************
 * libwiigui
 *
 * Tantric 2009
 * giantpune 2011
 *
 * gui_image.cpp
 *
 * GUI class definitions
 ***************************************************************************/

#include <pngdec/loadpng.h>
#include <stdio.h>
#include <string.h>

#include "gui.h"
#include "rsxmem.h"
#include "utils.h"

/**
 * Constructor for the GuiImageData class.
 */
GuiImageData::GuiImageData( const u8 * img, const u32 len )
{
	data = NULL;
	width = 0;
	height = 0;
	rsxOffset = 0;
	wpitch = 0;

	if( !img || len < 4 )
		return;

	//load png
	if( *(u32*)img == 0x89504e47 )//png magic word
	{
		//printf("GuiImageData() loading as PNG\n");
		PngDatas pd;

		pd.png_size = len;
		pd.png_in   = (void *)img;

		// load PNG from memory
		LoadPNG( &pd, NULL );

		if( !pd.bmp_out )
			return;

		//allocate memory in rsx buffer
		u32 texSize = pd.wpitch * pd.height;
		data = RsxMem::Alloc( texSize, &rsxOffset );
		if( !data )
		{
			free( pd.bmp_out );
			return;
		}
		//printf("GuiImageData(): data: %p, offset: %08x  size: %08x\n", data, rsxOffset, texSize );

		//copy bmp to rsx memory
		memcpy( data, pd.bmp_out, texSize );

		//hexdump( pd.bmp_out, 0x20 );
		free( pd.bmp_out );

		width = pd.width;
		height = pd.height;
        wpitch = pd.wpitch;
	}
}

GuiImageData::~GuiImageData()
{
	if( data )
	{
		RsxMem::Free( data );
		data = NULL;
	}
}

u8 * GuiImageData::GetImage()
{
	return data;
}

int GuiImageData::GetWidth()
{
	return width;
}

int GuiImageData::GetHeight()
{
	return height;
}

u32 GuiImageData::GetRsxTexOffset()
{
	return rsxOffset;
}

u32 GuiImageData::GetWPitch()
{
	return wpitch;
}

/**
 * Constructor for the GuiImage class.
 */
GuiImage::GuiImage()
{
	image = NULL;
	width = 0;
	height = 0;
	imageangle = 0;
	tile = -1;
	stripe = 0;
	imgType = IMAGE_DATA;
	rsxOffset = 0;
	wpitch = 0;
}

GuiImage::GuiImage( GuiImageData * img )
{
	image = NULL;
	width = 0;
	height = 0;
	rsxOffset = 0;
	wpitch = 0;
	if(img)
	{
		image = img->GetImage();
		width = img->GetWidth();
		height = img->GetHeight();
		rsxOffset = img->GetRsxTexOffset();
		wpitch = img->GetWPitch();
	}
	imageangle = 0;
	tile = -1;
	stripe = 0;
	imgType = IMAGE_DATA;

}

GuiImage::GuiImage( u8 * img, int w, int h )
{
	image = img;
	width = w;
	height = h;
	imageangle = 0;
	tile = -1;
	stripe = 0;
	wpitch = 0;
	rsxOffset = RsxMem::Offset( img );
	imgType = IMAGE_TEXTURE;
}

GuiImage::GuiImage(int w, int h, u32 c)
{
    width = w;
    height = h;
    imageangle = 0;
    tile = -1;
    stripe = 0;
    imgType = IMAGE_COLOR;
    image = NULL;
    color = c;
}

/**
 * Destructor for the GuiImage class.
 */
GuiImage::~GuiImage()
{
    //if( imgType == IMAGE_COLOR && image )
    //	RsxMem::Free( image );
}

u8 * GuiImage::GetImage()
{
	return image;
}

void GuiImage::SetImage( GuiImageData * img )
{
	image = NULL;
	width = 0;
	height = 0;
	rsxOffset = 0;
	wpitch = 0;
	if(img)
	{
		image = img->GetImage();
		width = img->GetWidth();
		height = img->GetHeight();
		rsxOffset = img->GetRsxTexOffset();
		wpitch = img->GetWPitch();
	}
	imgType = IMAGE_DATA;
}

void GuiImage::SetImage(u8 * img, int w, int h)
{
	image = img;
	width = w;
	height = h;
	imgType = IMAGE_TEXTURE;
	rsxOffset = RsxMem::Offset( img );
}

void GuiImage::SetAngle(float a)
{
	imageangle = a;
}

void GuiImage::SetTile(int t)
{
	tile = t;
}
/*
GXColor GuiImage::GetPixel(int x, int y)
{
	if(!image || this->GetWidth() <= 0 || x < 0 || y < 0)
		return (GXColor){0, 0, 0, 0};

	u32 offset = (((y >> 2)<<4)*this->GetWidth()) + ((x >> 2)<<6) + (((y%4 << 2) + x%4 ) << 1);
	GXColor color;
	color.a = *(image+offset);
	color.r = *(image+offset+1);
	color.g = *(image+offset+32);
	color.b = *(image+offset+33);
	return color;
}

void GuiImage::SetPixel(int x, int y, GXColor color)
{
	if(!image || this->GetWidth() <= 0 || x < 0 || y < 0)
		return;

	u32 offset = (((y >> 2)<<4)*this->GetWidth()) + ((x >> 2)<<6) + (((y%4 << 2) + x%4 ) << 1);
	*(image+offset) = color.a;
	*(image+offset+1) = color.r;
	*(image+offset+32) = color.g;
	*(image+offset+33) = color.b;
}

void GuiImage::SetStripe(int s)
{
	stripe = s;
}

void GuiImage::ColorStripe(int shift)
{
	int x, y;
	GXColor color;
	int alt = 0;

	for(y=0; y < this->GetHeight(); y++)
	{
		if(y % 3 == 0)
			alt ^= 1;

		for(x=0; x < this->GetWidth(); x++)
		{
			color = GetPixel(x, y);

			if(alt)
			{
				if(color.r < 255-shift)
					color.r += shift;
				else
					color.r = 255;
				if(color.g < 255-shift)
					color.g += shift;
				else
					color.g = 255;
				if(color.b < 255-shift)
					color.b += shift;
				else
					color.b = 255;

				color.a = 255;
			}
			else
			{
				if(color.r > shift)
					color.r -= shift;
				else
					color.r = 0;
				if(color.g > shift)
					color.g -= shift;
				else
					color.g = 0;
				if(color.b > shift)
					color.b -= shift;
				else
					color.b = 0;

				color.a = 255;
			}
			SetPixel(x, y, color);
		}
	}
	int len = width*height*4;
	if(len%32) len += (32-len%32);
	DCFlushRange(image, len);
}

void GuiImage::Grayscale()
{
	GXColor color;
	u32 offset, gray;

	for (int x = 0; x < width; x++)
	{
		for (int y = 0; y < height; y++)
		{
			offset = (((y >> 2) << 4) * this->GetWidth()) + ((x >> 2) << 6)
					+ (((y % 4 << 2) + x % 4) << 1);
			color.r = *(image + offset + 1);
			color.g = *(image + offset + 32);
			color.b = *(image + offset + 33);

			gray = (77 * color.r + 150 * color.g + 28 * color.b) / 255;

			*(image + offset + 1) = gray;
			*(image + offset + 32) = gray;
			*(image + offset + 33) = gray;
		}
	}
	int len = width*height*4;
	if(len%32) len += (32-len%32);
	DCFlushRange(image, len);
}
*/
/**
 * Draw the button on screen
 */
void GuiImage::Draw()
{
    if( ( !image && imgType != IMAGE_COLOR )
        || !this->IsVisible()
        || tile == 0 )
	{
        //printf( "GuiImage::Draw() abort: %p, %u, %i\n", image, this->IsVisible(), tile );
		return;
	}

	int currLeft = this->GetLeft();
    float currScale = this->GetScale();


    if( scale != 1 )
        currLeft = currLeft - width/2 + (width*scale)/2;

	if(tile > 0)
	{
		for(int i=0; i<tile; i++)
            MenuDrawImage( rsxOffset, width, height, wpitch, currLeft + width * i, this->GetTop(), 0, imageangle, this->GetAlpha(), currScale );
            //Menu_DrawImg(currLeft+width*i, this->GetTop(), width, height, image, imageangle, currScale, currScale, this->GetAlpha());
	}
	else
    {
		// temporary (maybe), used to correct offset for scaled images
        //if( scale != 1 )
            //currLeft = currLeft - ( ( width + ( width * scale ) ) / 2 );
            //currLeft = currLeft - width/2 + (width*scale)/2;

		//Menu_DrawImg(currLeft, this->GetTop(), width, height, image, imageangle, currScale, currScale, this->GetAlpha());

        if( imgType == IMAGE_COLOR )
        {
            MenuDrawRectangle( ( ( color & 0xffffff00 ) | ( this->GetAlpha() & 0xff ) ), width, height, currLeft, this->GetTop(), 0, true, imageangle, currScale );
        }
        else
        {
            MenuDrawImage( rsxOffset, width, height, wpitch, currLeft, this->GetTop(), 0, imageangle, this->GetAlpha(), currScale );
        }
    }

    /*if(stripe > 0)
		for(int y=0; y < this->GetHeight(); y+=6)
			Menu_DrawRectangle(currLeft,this->GetTop()+y,this->GetWidth(),3,(GXColor){0, 0, 0, stripe},1);*/

	this->UpdateEffects();
}
