/****************************************************************************
 * this class is based off libfont by hermes
 * giantpune 2011
 ***************************************************************************/


#include <stdio.h>

#include "guifont.h"
#include "rsxmem.h"
#include "utils.h"

GuiFont::GuiFont( u8* mem, u32 len, u16 fChar, u16 lChar, u16 size )
{

	//init variables
	pointer = NULL;
	rsxOffset = 0;
	firstChar = 0;
	lastChar = 0;
	rsxBytesPerChar = 0;
	bh = 8;
	storedSize = 0;

	if( fChar >= FONT_MAX_CHARS || fChar > lChar )
		return;

	if( lChar >= FONT_MAX_CHARS )
		lChar = FONT_MAX_CHARS - 1;

	firstChar = fChar;
	lastChar = lChar;

	//allocate memory for texture
	u32 texSize = GetTextureSize( lChar - fChar, size );
	pointer = RsxMem::Alloc( texSize, &rsxOffset );
	if( !pointer )
	{
		printf( "GuiFont::GuiFont() failed to allocate memory\n" );
		return;
	}

	//load font with freetype
	FT_Library freetype;

	FT_Init_FreeType( &freetype );
	if( FT_New_Memory_Face( freetype, (const FT_Byte*)mem, len, 0, &face ) )
		return;

	firstChar = fChar;
	lastChar = lChar;
	storedSize = size;

	CreateTexture( size );
	FT_Done_FreeType( freetype );
}

GuiFont::~GuiFont()
{
	if( pointer )
		RsxMem::Free( pointer );
}

u32 GuiFont::GetTextureSize( u32 numChars, u16 size )
{
	u32 ret = 0;

	u32 xx = ( size * size * 2 );
	if( !numChars )
		numChars = 1;

	for( u32 n = 0; n < numChars; n++ )
	{
		ret += xx;
		ret = ((((long) ret) + 15) & ~15);
	}
	return ret;
}

void GuiFont::CreateTexture( u32 size )
{
	u8 i;
	u8 *font;
	u8* texture = pointer;
	static u8 letter_bitmap[257 * 256];
	u8 bits_per_pixel = 8;

	for( int n = 0; n < FONT_MAX_CHARS; n++ )
	{
		fw[ n ] = 0;
		fy[ n ] = 0;
	}

	//in case somebody tries to load a font with 1 letter
	u16 lC = lastChar;
	if( lC == firstChar )
		lC++;

	for( u32 n = firstChar; n <= lC; n++ )
	{

		//short hh = size;
		short hh = size;

		font = (u8*)&letter_bitmap;

		fw[ n ] = size;

		TTF_to_Bitmap( (u8) (n & 0xff), (u8*)(&letter_bitmap), &fw[ n ], &hh,  &fy[ n ] );

		//printf("callback done.  \'%c\', fw: %i  hh: %i  fy: %i\n", n, fw[n], hh, fy[n] );

		// letter background correction
		if( ( hh + fy[ n ] ) > bh )
			bh = hh + fy[ n ];

		if( n == (u32)(firstChar + 1) )
			rsxBytesPerChar = RsxMem::Offset( texture ) - rsxOffset;

		if( n == lastChar )
			break;


		//printf("%c %p\n", n, texture );
		//void *ppp = (void*)texture;
		for( u32 a = 0; a < size; a++ )
		{
			for( u32 b = 0; b < size; b++ )
			{
				i = font[ ( b * bits_per_pixel ) / 8 ];

				i >>= ( b & ( 7 / bits_per_pixel ) ) * bits_per_pixel;

				i = ( i & ( ( 1 << bits_per_pixel ) - 1 ) ) * 255 / ( ( 1 << bits_per_pixel ) - 1 );

				if( i )
				{	//TINY3D_TEX_FORMAT_A4R4G4B4
					i>>=4;
					*((u16 *)texture) = ( i<<12 ) | 0xfff;
				}
				else
				{
					texture[ 0 ] = texture[ 1 ] = 0x0; //texture[2] = 0x0;
					//texture[3] = 0x0; // alpha
				}
				texture += 2;
			}
			font += ( size * bits_per_pixel ) / 8;
		}
		//hexdump( ppp, 0x500 );
		texture = (u8 *)((((long) texture) + 15) & ~15);

	}
	//hexdump( (void*)&letter_bitmap, 257 * 256 );
	//return texture;
}

void GuiFont::DrawText( float x, float y, float z, wchar_t *text, u32 rgba, u32 size, uint16_t textStyle )
{
    if( !text )
        return;
	int len = wcslen( text );
	DrawText( x, y, z, text, len, rgba, size, textStyle );
}

void GuiFont::DrawText( float x, float y, float z, wchar_t *text, u32 len, u32 rgba, u32 size, uint16_t textStyle )
{
    if( !text )
        return;
	for( u32 i = 0; i < len; i++ )
	{
		if( text[ i ] > FONT_MAX_CHARS )
			break;
		DrawCh( x, y, z, text[ i ], rgba, size );
		x += (float) ( ( fw[ text[ i ] ] * size ) / storedSize );
	}
}

void GuiFont::DrawCh( float x, float y, float z, u16 chr, u32 rgba, u32 size )
{
	//printf("DrawChar( %.2f, %.2f, %.2f, %04x, %08x, %u )\n", x, y, z, chr, rgba, size );
	if( chr < firstChar || chr > lastChar )
	{
		//printf("GuiFont::DrawCh() out of range");
		return;
	}

	y += (float) ( ( fy[ chr ] * size ) / storedSize );

	float dx  = x + size,
	dy = y + size;

	// Load sprite texture
	tiny3d_SetTexture( 0, rsxOffset + rsxBytesPerChar * ( chr - firstChar ), storedSize, storedSize, storedSize * 2, TINY3D_TEX_FORMAT_A4R4G4B4, 1 );

	tiny3d_SetPolygon( TINY3D_QUADS );

	tiny3d_VertexPos( x, y, z );
	tiny3d_VertexColor( rgba );
	tiny3d_VertexTexture( 0.0f, 0.0f );

	tiny3d_VertexPos( dx, y, z );
	tiny3d_VertexTexture( 0.95f, 0.0f );

	tiny3d_VertexPos( dx, dy, z );
	tiny3d_VertexTexture( 0.95f, 0.95f );

	tiny3d_VertexPos( x, dy, z );
	tiny3d_VertexTexture( 0.0f, 0.95f );

	tiny3d_End();
}

void GuiFont::TTF_to_Bitmap( u16 chr, u8 * bitmap, short *w, short *h, short *y_correction )
{
	FT_Set_Pixel_Sizes( face, (*w), (*h) );

	FT_GlyphSlot slot = face->glyph;

	memset( bitmap, 0, (*w) * (*h) );

	if( FT_Load_Char(face, chr, FT_LOAD_RENDER ) )
	{
		(*w) = 0;
		return;
	}

	int n, m, ww;

	*y_correction = (*h) - 1 - slot->bitmap_top;

	ww = 0;

	for( n = 0; n < slot->bitmap.rows; n++ )
	{
		for ( m = 0; m < slot->bitmap.width; m++ )
		{
			if( m >= (*w) || n >= (*h) )
				continue;

			bitmap[ m ] = (u8) slot->bitmap.buffer[ ww + m ];
		}
		bitmap += *w;
		ww += slot->bitmap.width;
	}

	*w = ( ( slot->advance.x + 31 ) >> 6 ) + ( ( slot->bitmap_left < 0 ) ? -slot->bitmap_left : 0 );
	*h = slot->bitmap.rows;
}


u32 GuiFont::TextWidth( wchar_t *text, u32 size )
{
    if( !text )
        return 0;
	int len = wcslen( text );
	return TextWidth( text, len, size );
}

u32 GuiFont::TextWidth( wchar_t *text, u32 len, u32 size )
{
    if( !text )
        return 0;
	u32 ret = 0;
	for( u32 i = 0; i < len; i++ )
	{
		if( text[ i ] > FONT_MAX_CHARS )
			break;
		ret += (float) ( ( fw[ text[ i ] ] * size ) / storedSize );
	}
	return ret;
}
