/*
             LUFA Library
     Copyright (C) Dean Camera, 2010.

  dean [at] fourwalledcubicle [dot] com
      www.fourwalledcubicle.com
*/

/*
  Copyright 2010  Dean Camera (dean [at] fourwalledcubicle [dot] com)

  Permission to use, copy, modify, distribute, and sell this
  software and its documentation for any purpose is hereby granted
  without fee, provided that the above copyright notice appear in
  all copies and that both that the copyright notice and this
  permission notice and warranty disclaimer appear in supporting
  documentation, and that the name of the author not be used in
  advertising or publicity pertaining to distribution of the
  software without specific, written prior permission.

  The author disclaim all warranties with regard to this
  software, including all implied warranties of merchantability
  and fitness.  In no event shall the author be liable for any
  special, indirect or consequential damages or any damages
  whatsoever resulting from loss of use, data or profits, whether
  in an action of contract, negligence or other tortious action,
  arising out of or in connection with the use or performance of
  this software.
*/

/** \file
 *
 *  Header file for ISPTarget.c.
 */

#ifndef _ISP_TARGET_
#define _ISP_TARGET_

	/* Includes: */
		#include <avr/io.h>
		#include <avr/pgmspace.h>
		#include <util/delay.h>

		#include <LUFA/Drivers/USB/USB.h>
		#include <LUFA/Drivers/Peripheral/SPI.h>

		#include "../V2ProtocolParams.h"

	/* Preprocessor Checks: */
		#if ((BOARD == BOARD_XPLAIN) || (BOARD == BOARD_XPLAIN_REV1))
			#undef ENABLE_ISP_PROTOCOL

			#if !defined(ENABLE_XPROG_PROTOCOL)
				#define ENABLE_XPROG_PROTOCOL
			#endif
		#endif

	/* Macros: */
		/** Low level device command to issue an extended FLASH address, for devices with other 128KB of FLASH. */
		#define LOAD_EXTENDED_ADDRESS_CMD     0x4D

		/** Macro to convert an ISP frequency to a number of timer clock cycles for the software SPI driver */
		#define TIMER_COMP(freq) ((((F_CPU / 8) / freq) / 2) - 1)

	/* External Variables: */
		extern bool HardwareSPIMode;

	/* Function Prototypes: */
		void    ISPTarget_Init(void);
		void    ISPTarget_ShutDown(void);
		uint8_t ISPTarget_TransferSoftSPIByte(const uint8_t Byte);
		void    ISPTarget_ChangeTargetResetLine(const bool ResetTarget);
		uint8_t ISPTarget_WaitWhileTargetBusy(void);
		void    ISPTarget_LoadExtendedAddress(void);
		uint8_t ISPTarget_WaitForProgComplete(const uint8_t ProgrammingMode,
		                                      const uint16_t PollAddress,
		                                      const uint8_t PollValue,
		                                      const uint8_t DelayMS,
		                                      const uint8_t ReadMemCommand);

	/* Inline Functions: */
		/** Sends a byte of ISP data to the attached target, using the appropriate SPI hardware or
		 *  software routines depending on the selected ISP speed.
		 *
		 *  \param[in] Byte  Byte of data to send to the attached target
		 */
		static inline void ISPTarget_SendByte(const uint8_t Byte)
		{
			if (HardwareSPIMode)
			  SPI_SendByte(Byte);
			else
			  ISPTarget_TransferSoftSPIByte(Byte);
		}

		/** Receives a byte of ISP data from the attached target, using the appropriate
		 *  SPI hardware or software routines depending on the selected ISP speed.
		 *
		 *  \return Received byte of data from the attached target
		 */
		static inline uint8_t ISPTarget_ReceiveByte(void)
		{
			if (HardwareSPIMode)
			  return SPI_ReceiveByte();
			else
			  return ISPTarget_TransferSoftSPIByte(0x00);
		}

		/** Sends and receives a byte of ISP data to and from the attached target, using the
		 *  appropriate SPI hardware or software routines depending on the selected ISP speed.
		 *
		 *  \param[in] Byte  Byte of data to send to the attached target
		 *
		 *  \return Received byte of data from the attached target
		 */
		static inline uint8_t ISPTarget_TransferByte(const uint8_t Byte)
		{
			if (HardwareSPIMode)
			  return SPI_TransferByte(Byte);
			else
			  return ISPTarget_TransferSoftSPIByte(Byte);
		}

#endif

